package oauth

import (
	"context"
	"crypto/x509"
	"encoding/pem"
	"github.com/dghubble/oauth1"
	"github.com/pkg/errors"
	"net/http"
	"strings"
)

// GenerateOAuthClient generates an oauth 1 client with the supplied config
func GenerateOAuthClient(privateKey, consumerKey, accessToken, accessTokenSecret string) (*http.Client, error) {
	ctx := context.Background()
	var oauthConfig *oauth1.Config

	if privateKey == "" {
		return nil, errors.New("empty private key")
	} else if keyDERBlock, _ := pem.Decode([]byte(privateKey)); keyDERBlock == nil {
		return nil, errors.New("error decoding key PEM block")
	} else if !(keyDERBlock.Type == "PRIVATE KEY" || strings.HasSuffix(keyDERBlock.Type, " PRIVATE KEY")) {
		return nil, errors.Errorf("unexpected key DER block type %v", keyDERBlock.Type)
	} else if privateKey, err := x509.ParsePKCS1PrivateKey(keyDERBlock.Bytes); err != nil {
		return nil, errors.Wrap(err, "parsing PKCS1 private key")
	} else {
		oauthConfig = &oauth1.Config{
			ConsumerKey: consumerKey,
			Signer:      &oauth1.RSASigner{PrivateKey: privateKey},
		}
	}

	httpClient := oauth1.NewClient(ctx, oauthConfig, oauth1.NewToken(accessToken, accessTokenSecret))
	return httpClient, nil
}
