package oauth

import (
	"github.com/stretchr/testify/assert"
	"strings"
	"testing"
)

func TestOAuthConfig(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	consumerKey := "TestConsumerKey"
	accessToken := "TestAccessToken"
	accessTokenSecret := "TestAccessTokenSecret"
	privateKey := ""

	// Test with empty private key
	_, err := GenerateOAuthClient(privateKey, consumerKey, accessToken, accessTokenSecret)
	a.NotNil(err)
	a.EqualValues("empty private key", err.Error())

	// Test with private key that isn't PEM
	privateKey = "BLAHBLAH"
	_, err = GenerateOAuthClient(privateKey, consumerKey, accessToken, accessTokenSecret)
	a.NotNil(err)
	a.EqualValues("error decoding key PEM block", err.Error())

	/*
		WOOP WOOP ATTENTION SECURITY AUDIT PERSONNEL!
		These are test keys that were generated purely for the purposes of this test and are not in use anywhere but here
		WOOP WOOP ATTENTION SECURITY AUDIT PERSONNEL!
	*/

	// Test with key that isn't private -
	privateKey = `-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA3Ku8+YpQrQ6L0nx4Iw8Y
Wim1dPC/JlEsdYFtEjrwoKKP1ozmVdY3nTs6Wj6hHL3PEVqCT2pSIiMJy0OHra9P
e1HIHZ8SSZ1ClhtpGi24bWtdb03qnNJn2Sm7XevwrcU/b5KMdIrNMo8uFKwXsncR
J4tJESkCdkrhROgAD1q3sK2+MHAT2/ULTXQZpfKdCfXdVjR/75v+zpJFoCiE/uLj
sUcsz67M/U4igY5GcgxL9lFVpEUHN/gUEg0XfZuOq8n/RPZHrSmOBcl455JfE+m9
AN1AZ7tYRDfl3aziwwDt5lqaVuU3D3YKX445BLOVpUKNQ3pDVWeQPRvpli60Pb3a
AwIDAQAB
-----END PUBLIC KEY-----`
	_, err = GenerateOAuthClient(privateKey, consumerKey, accessToken, accessTokenSecret)
	a.NotNil(err)
	a.EqualValues("unexpected key DER block type PUBLIC KEY", err.Error())

	/*
		WOOP WOOP ATTENTION SECURITY AUDIT PERSONNEL!
		These are test keys that were generated purely for the purposes of this test and are not in use anywhere but here
		WOOP WOOP ATTENTION SECURITY AUDIT PERSONNEL!
	*/

	// Test with malformed private key
	privateKey = `-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEA3Ku8+YpQrQ6L0nx4Iw8YWim1dPC/JlEsdYFtEjrwoKKP1ozm
VdY3nTs6Wj6hHL3PEVqCT2pSIiMJy0OHra9Pe1HIHZ8SSZ1ClhtpGi24bWtdb03q
nNJn2Sm7XevwrcU/b5KMdIrNMo8uFKwXsncRJ4tJESkCdkrhROgAD1q3sK2+MHAT
2/ULTXQZpfKdCfXdVjR/75v+zpJFoCiE/uLjsUcsz67M/U4igY5GcgxL9lFVpEUH
N/gUEg0XfZuOq8n/RPZHrSmOBcl455JfE+m9AN1AZ7tYRDfl3aziwwDt5lqaVuU3
D3YKX445BLOVpUKNQ3pDVWeQPRvpli60Pb3aAwIDAQABAoIBADz/rCk+trZfwv7Y
MU1fqtehl+tPcUdiQOvqbwAPEt0UkrY4FKO9p3nrGWJA1cBgJ5Egd80EEtgk31w0
upikUVBBdxPZGPEIa5Oa2B7IbKayew19eFEJSHObOZPo2Irwb2ykEfdVNwu20IIe
YB6RE0jfE7+OmoOgh0zI5NtZtFU97g4QCdGK6XoeqpyGbPKX/ADd0Gv8BZbCCBTu
ww9ubNoYNZh+KG8lZkWV6Gp17dk0CKsmJKoSYmCKVWiOAsg+Aty4KSmzbi/1sOFb
dqIdqKfmr0xMPJwIXhNX3ur+Mwffc6ZcKRA7c+dtDcNIYOktUroPye5ekWfgx715
8zVzgKECgYEA/w1DLvXZBxigxIiQIRl2RqS3UdB/t571FrqUIG+Q+2OCEB14kiLT
OJh7G7J+DSbgN6HOwz2OC3m9xJhr4QLJNT/50l1DiCIkPjSpMF/LSNT7duvZN9Mw
Ok5XJjccHKW4x/U8l5hIf878ugggWMSczWu0QV+8qjX2o2dDofbHco0CgYEA3X3B
N2cIIjTS0Dygz1r6RFMuI/prcmNW4gm3GVniGQVKWFHXQWRRWRvU2TwL1rRIkAgE
p1htlpmZkdKnhRmlO16I86zClWnzG7yg1MAnklVhX9qRkPptcPKtfL8IZjjaaOGF
IjORMTcTitofE7SjxbrpDZK+EEYVSHgJh0qnos7FgYAG62MxOPZEXQSvkEzouPoe
fIWEsLVmU+euUC2Im75BM7fpnAMTY+J7HJfqbnNBBBb9K2Sio17hLNJbLQ9wMVUo
Hq42aNVWMibtkRwt9ctcFHsmBMwF2bLrQgjXPpOCq3RR6HNWYYlF/mWY3nOSNPNU
F2HyjBhgPcqNMFYDZOCQWQKBgQCNO0vCBQu1rBMkvxQXRD7C5HtOa7apuNhEx8cd
/idipvHyATWNt+b7N/jwEA7IeCqw1LwK7SLr3cODrVwuTd0DT1NqdaWZ2nYvIIc5
QdG5yIGPunIJjPXOCKBBJBq5+9Sg/fSDi3gwdpsadd1NFmk7guRqSECzZczaadhB
kYP1FwKBgCFCEa5sGjMmZLxMbMzHtMkOX0FLFhDWGsIZiRzMkZEbNa5HgNvYFZZP
fq+ztbxQjjO+io0WtEDeFgeI8hUb413aYua/Eq5AXeQb1SFiTysPbAAAun8P90Yf
f/ShDF4bqy5Vt7/ywvos/zqnw+6aDmZR43e5il4kMHUIxyA7Dq8f
-----END RSA PRIVATE KEY-----`
	_, err = GenerateOAuthClient(privateKey, consumerKey, accessToken, accessTokenSecret)
	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "parsing PKCS1 private key:"), "error message should start with parsing PKCS1 private key: - got `%v`", err.Error())

	/*
		WOOP WOOP ATTENTION SECURITY AUDIT PERSONNEL!
		These are test keys that were generated purely for the purposes of this test and are not in use anywhere but here
		WOOP WOOP ATTENTION SECURITY AUDIT PERSONNEL!
	*/

	// Test with valid private key
	privateKey = `-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEA3Ku8+YpQrQ6L0nx4Iw8YWim1dPC/JlEsdYFtEjrwoKKP1ozm
VdY3nTs6Wj6hHL3PEVqCT2pSIiMJy0OHra9Pe1HIHZ8SSZ1ClhtpGi24bWtdb03q
nNJn2Sm7XevwrcU/b5KMdIrNMo8uFKwXsncRJ4tJESkCdkrhROgAD1q3sK2+MHAT
2/ULTXQZpfKdCfXdVjR/75v+zpJFoCiE/uLjsUcsz67M/U4igY5GcgxL9lFVpEUH
N/gUEg0XfZuOq8n/RPZHrSmOBcl455JfE+m9AN1AZ7tYRDfl3aziwwDt5lqaVuU3
D3YKX445BLOVpUKNQ3pDVWeQPRvpli60Pb3aAwIDAQABAoIBADz/rCk+trZfwv7Y
MU1fqtehl+tPcUdiQOvqbwAPEt0UkrY4FKO9p3nrGWJA1cBgJ5Egd80EEtgk31w0
upikUVBBdxPZGPEIa5Oa2B7IbKayew19eFEJSHObOZPo2Irwb2ykEfdVNwu20IIe
YB6RE0jfE7+OmoOgh0zI5NtZtFU97g4QCdGK6XoeqpyGbPKX/ADd0Gv8BZbCCBTu
ww9ubNoYNZh+KG8lZkWV6Gp17dk0CKsmJKoSYmCKVWiOAsg+Aty4KSmzbi/1sOFb
dqIdqKfmr0xMPJwIXhNX3ur+Mwffc6ZcKRA7c+dtDcNIYOktUroPye5ekWfgx715
8zVzgKECgYEA/w1DLvXZBxigxIiQIRl2RqS3UdB/t571FrqUIG+Q+2OCEB14kiLT
OJh7G7J+DSbgN6HOwz2OC3m9xJhr4QLJNT/50l1DiCIkPjSpMF/LSNT7duvZN9Mw
Ok5XJjccHKW4x/U8l5hIf878ugggWMSczWu0QV+8qjX2o2dDofbHco0CgYEA3X3B
N2cIIjTS0Dygz1r6RFMuI/prcmNW4gm3GVniGQVKWFHXQWRRWRvU2TwL1rRIkAgE
p1htlpmZkdKnhRmlO16I86zClWnzG7yg1MAnklVhX9qRkPptcPKtfL8IZjjaaOGF
IjORMTcTitofE7SjxbrpDZK+EEYVSHgJh0qnos8CgYAG62MxOPZEXQSvkEzouPoe
fIWEsLVmU+euUC2Im75BM7fpnAMTY+J7HJfqbnNBBBb9K2Sio17hLNJbLQ9wMVUo
Hq42aNVWMibtkRwt9ctcFHsmBMwF2bLrQgjXPpOCq3RR6HNWYYlF/mWY3nOSNPNU
F2HyjBhgPcqNMFYDZOCQWQKBgQCNO0vCBQu1rBMkvxQXRD7C5HtOa7apuNhEx8cd
/idipvHyATWNt+b7N/jwEA7IeCqw1LwK7SLr3cODrVwuTd0DT1NqdaWZ2nYvIIc5
QdG5yIGPunIJjPXOCKBBJBq5+9Sg/fSDi3gwdpsadd1NFmk7guRqSECzZczaadhB
kYP1FwKBgCFCEa5sGjMmZLxMbMzHtMkOX0FLFhDWGsIZiRzMkZEbNa5HgNvYFZZP
fq+ztbxQjjO+io0WtEDeFgeI8hUb413aYua/Eq5AXeQb1SFiTysPbAAAun8P90Yf
f/ShDF4bqy5Vt7/ywvos/zqnw+6aDmZR43e5il4kMHUIxyA7Dq8f
-----END RSA PRIVATE KEY-----`
	client, err := GenerateOAuthClient(privateKey, consumerKey, accessToken, accessTokenSecret)
	a.NotNil(client)
	a.Nil(err)
}
