package backend

import (
	logging "code.justin.tv/event-engineering/goldengate/pkg/logging/backend"
	"encoding/json"
	"fmt"
	pdEvents "github.com/marcw/pagerduty"
	"io"
	"net"
	"net/http"
	//"net/url"
	"runtime"
	"time"
)

const (
	apiEndpoint = "https://api.pagerduty.com"
)

// Client redefines goPd.Client to allow custom methods and easier data mocking for tests.
// If this interface changes, counterfeiter must be re-run using `make mocks`
type Client interface {
	ListOnCalls(escalationPolicyID string) (*ListOnCallsResponse, error)
	ListTeamUsers(teamID string) (*ListUsersResponse, error)
	SubmitEvent(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error)
}

type client struct {
	authToken  string
	httpClient *http.Client
	logger     logging.Logger
}

// New creates a new PagerDuty backend
func New(authToken string, logger logging.Logger) Client {
	httpClient := &http.Client{
		Transport: &http.Transport{
			Proxy: http.ProxyFromEnvironment,
			DialContext: (&net.Dialer{
				Timeout:   30 * time.Second,
				KeepAlive: 30 * time.Second,
			}).DialContext,
			MaxIdleConns:          10,
			IdleConnTimeout:       60 * time.Second,
			TLSHandshakeTimeout:   10 * time.Second,
			ExpectContinueTimeout: 1 * time.Second,
			MaxIdleConnsPerHost:   runtime.GOMAXPROCS(0) + 1,
		},
	}

	return &client{
		authToken:  authToken,
		httpClient: httpClient,
		logger:     logger,
	}
}

func (c *client) get(path string) (*http.Response, error) {
	return c.do("GET", path, nil)
}

func (c *client) do(method, path string, body io.Reader) (*http.Response, error) {
	endpoint := apiEndpoint + path
	req, err := http.NewRequest(method, endpoint, body)
	if err != nil {
		return nil, err
	}

	req.Header.Set("Accept", "application/vnd.pagerduty+json;version=2")
	req.Header.Set("Content-Type", "application/json")
	req.Header.Set("Authorization", "Token token="+c.authToken)

	resp, err := c.httpClient.Do(req)
	return c.checkResponse(resp, err)
}

func (c *client) checkResponse(resp *http.Response, err error) (*http.Response, error) {
	if err != nil {
		return resp, fmt.Errorf("Error calling the API endpoint: %v", err)
	}
	if 199 >= resp.StatusCode || 300 <= resp.StatusCode {
		var eo *errorObject
		var getErr error
		if eo, getErr = c.getErrorFromResponse(resp); getErr != nil {
			return resp, fmt.Errorf("Response did not contain formatted error: %s. HTTP response code: %v. Raw response: %+v", getErr, resp.StatusCode, resp)
		}
		return resp, fmt.Errorf("Failed call API endpoint. HTTP response code: %v. Error: %v", resp.StatusCode, eo)
	}
	return resp, nil
}

func (c *client) decodeJSON(resp *http.Response, payload interface{}) error {
	defer func() {
		err := resp.Body.Close()
		if err != nil {
			c.logger.Error("Error closing response body", err)
		}
	}()

	decoder := json.NewDecoder(resp.Body)
	return decoder.Decode(payload)
}

func (c *client) getErrorFromResponse(resp *http.Response) (*errorObject, error) {
	var result map[string]errorObject
	if err := c.decodeJSON(resp, &result); err != nil {
		return nil, fmt.Errorf("Could not decode JSON response: %v", err)
	}
	s, ok := result["error"]
	if !ok {
		return nil, fmt.Errorf("JSON response does not have error field")
	}
	return &s, nil
}

func (c *client) ListOnCalls(escalationPolicyID string) (*ListOnCallsResponse, error) {
	resp, err := c.get(fmt.Sprintf("/oncalls?time_zone=UTC&include[]=users&escalation_policy_ids[]=%v", escalationPolicyID))
	if err != nil {
		return nil, err
	}

	onCallsResponse := &ListOnCallsResponse{}
	err = c.decodeJSON(resp, onCallsResponse)
	if err != nil {
		return nil, err
	}

	return onCallsResponse, nil
}

func (c *client) ListTeamUsers(teamID string) (*ListUsersResponse, error) {
	resp, err := c.get(fmt.Sprintf("/users?include[]=contact_methods&team_ids[]=%v", teamID))
	if err != nil {
		return nil, err
	}

	usersResponse := &ListUsersResponse{}
	err = c.decodeJSON(resp, usersResponse)
	if err != nil {
		return nil, err
	}

	return usersResponse, nil
}

func (c *client) SubmitEvent(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
	return pdEvents.Submit(event)
}
