package pagerduty

import (
	logging "code.justin.tv/event-engineering/goldengate/pkg/logging/backend"
	"code.justin.tv/event-engineering/goldengate/pkg/pagerduty/backend"
	"fmt"
	pdEvents "github.com/marcw/pagerduty"
	"github.com/pkg/errors"
	"net/http"
)

// PhoneNumberType is the type string that specifies a ContactMethod is a phone number
const PhoneNumberType = "phone_contact_method"

// PhoneNumberReferenceType is the type string that specifies a ContactMethodReference is a phone number reference
const PhoneNumberReferenceType = "phone_contact_method_reference"

// Client is a Twilio client containing methods for use in the goldengate system
type Client struct {
	pd        backend.Client
	logger    logging.Logger
	teamUsers *teamUsersCache
}

// New creates a new PagerDuty client with the supplied backend
func New(client backend.Client, logger logging.Logger) *Client {
	return &Client{
		pd:     client,
		logger: logger,
	}
}

// TriggerEvent triggers an incident with the supplied parameters
func (c *Client) TriggerEvent(serviceKey, incidentKey, description string) error {
	event := pdEvents.NewTriggerEvent(serviceKey, description)

	event.IncidentKey = incidentKey
	resp, status, err := c.pd.SubmitEvent(event)

	if err != nil {
		return errors.Wrap(err, "Failed to submit PD event")
	}

	if status != http.StatusOK {
		return fmt.Errorf("Got non success status code %v with error %v", status, resp.Message)
	}

	return nil
}

// AckEvent acknowledges an incident with the supplied parameters
func (c *Client) AckEvent(serviceKey, incidentKey, description string) error {
	event := pdEvents.NewAcknowledgeEvent(serviceKey, description)

	event.IncidentKey = incidentKey
	resp, status, err := c.pd.SubmitEvent(event)

	if err != nil {
		return errors.Wrap(err, "Failed to submit PD event")
	}

	if status != http.StatusOK {
		return fmt.Errorf("Got non success status code %v with error %v", status, resp.Message)
	}

	return nil
}

// ResolveEvent rewolves an incident with the supplied parameters
func (c *Client) ResolveEvent(serviceKey, incidentKey, description string) error {
	event := pdEvents.NewResolveEvent(serviceKey, description)

	event.IncidentKey = incidentKey
	resp, status, err := c.pd.SubmitEvent(event)

	if err != nil {
		return errors.Wrap(err, "Failed to submit PD event")
	}

	if status != http.StatusOK {
		return fmt.Errorf("Got non success status code %v with error %v", status, resp.Message)
	}

	return nil
}

// GetOncallUsers retrieves all the phone numbers for users on call at the supplied escalation level for the supplied escalation policy
func (c *Client) GetOncallUsers(escalationPolicyID string, escalationLevel int) ([]*backend.OnCallUser, error) {
	resp, err := c.pd.ListOnCalls(escalationPolicyID)

	if err != nil {
		return nil, errors.Wrap(err, "Failed to get Oncall Users")
	}

	if resp.OnCalls == nil || len(resp.OnCalls) == 0 {
		return nil, errors.New("No Oncall Users returned")
	}

	users := make([]*backend.OnCallUser, 0)

	for _, oncall := range resp.OnCalls {
		if escalationLevel == 0 || oncall.EscalationLevel == escalationLevel {
			// The User object is populated even if the oncall is a schedule
			if oncall.User != nil {
				for _, cm := range oncall.User.ContactMethods {
					if cm.Type == PhoneNumberReferenceType {
						users = append(users, oncall.User)
					}
				}
			}
		}
	}

	return users, nil
}
