package pagerduty

import (
	loggingfakes "code.justin.tv/event-engineering/goldengate/pkg/logging/backend/backendfakes"
	"code.justin.tv/event-engineering/goldengate/pkg/pagerduty/backend"
	"code.justin.tv/event-engineering/goldengate/pkg/pagerduty/backend/backendfakes"
	pdEvents "github.com/marcw/pagerduty"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
	"strings"
	"testing"
	"time"
)

func TestTriggerEvent(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeClient := new(backendfakes.FakeClient)
	fakeLogger := new(loggingfakes.FakeLogger)
	client := New(fakeClient, fakeLogger)

	// Test successful event trigger
	fakeClient.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return &pdEvents.Response{
			Message: "SeemsGood",
		}, 200, nil
	}

	err := client.TriggerEvent("pd_service_key", "incident_1", "This is a description of the event")
	a.Nil(err)

	err = client.AckEvent("pd_service_key", "incident_1", "This is a description of the event")
	a.Nil(err)

	err = client.ResolveEvent("pd_service_key", "incident_1", "This is a description of the event")
	a.Nil(err)

	// Test error on event trigger
	fakeClient.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return nil, 500, errors.New("Some shit went wrong")
	}

	err = client.TriggerEvent("pd_service_key", "incident_1", "This is a description of the event")
	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "Failed to submit PD event:"), "error message should start with Failed to submit PD event: - got `%v`", err.Error())

	err = client.AckEvent("pd_service_key", "incident_1", "This is a description of the event")
	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "Failed to submit PD event:"), "error message should start with Failed to submit PD event: - got `%v`", err.Error())

	err = client.ResolveEvent("pd_service_key", "incident_1", "This is a description of the event")
	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "Failed to submit PD event:"), "error message should start with Failed to submit PD event: - got `%v`", err.Error())

	// Test bad request on event trigger
	fakeClient.SubmitEventStub = func(event *pdEvents.Event) (response *pdEvents.Response, statusCode int, err error) {
		return &pdEvents.Response{Message: "Bad Reqiest"}, 401, nil
	}

	err = client.TriggerEvent("pd_service_key", "incident_1", "This is a description of the event")
	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "Got non success status code"), "error message should start with Got non success status code - got `%v`", err.Error())

	err = client.AckEvent("pd_service_key", "incident_1", "This is a description of the event")
	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "Got non success status code"), "error message should start with Got non success status code - got `%v`", err.Error())

	err = client.ResolveEvent("pd_service_key", "incident_1", "This is a description of the event")
	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "Got non success status code"), "error message should start with Got non success status code - got `%v`", err.Error())
}

var onCalls = &backend.ListOnCallsResponse{
	OnCalls: []*backend.OnCall{
		&backend.OnCall{
			EscalationLevel: 1,
			User: &backend.OnCallUser{
				ID: "USER1",
				ContactMethods: []*backend.ContactMethodReference{
					&backend.ContactMethodReference{
						ID:   "CM1",
						Type: "phone_contact_method_reference",
					},
					&backend.ContactMethodReference{
						ID:   "CM2",
						Type: "email_contact_method_reference",
					},
				},
			},
		},
		&backend.OnCall{
			EscalationLevel: 1,
			User: &backend.OnCallUser{
				ID: "USER2",
				ContactMethods: []*backend.ContactMethodReference{
					&backend.ContactMethodReference{
						ID:   "CM3",
						Type: "phone_contact_method_reference",
					},
				},
			},
		},
		&backend.OnCall{
			EscalationLevel: 2,
			User: &backend.OnCallUser{
				ID: "USER3",
				ContactMethods: []*backend.ContactMethodReference{
					&backend.ContactMethodReference{
						ID:   "CM4",
						Type: "phone_contact_method_reference",
					},
				},
			},
		},
		&backend.OnCall{
			EscalationLevel: 3,
			User: &backend.OnCallUser{
				ID: "USER4",
				ContactMethods: []*backend.ContactMethodReference{
					&backend.ContactMethodReference{
						ID:   "CM5",
						Type: "email_contact_method_reference",
					},
				},
			},
		},
	},
}

var contactMethods = []*backend.ContactMethod{
	&backend.ContactMethod{
		ID:          "CM1",
		Type:        "phone_contact_method",
		Address:     "1234567890",
		CountryCode: 44,
	},
	&backend.ContactMethod{
		ID:      "CM2",
		Type:    "email_contact_method",
		Address: "example@example.com",
	},
	&backend.ContactMethod{
		ID:          "CM3",
		Type:        "phone_contact_method",
		Address:     "8593738930",
		CountryCode: 1,
	},
	&backend.ContactMethod{
		ID:          "CM4",
		Type:        "phone_contact_method",
		Address:     "0989897695",
		CountryCode: 31,
	},
}

func userInList(a string, list []*backend.OnCallUser) bool {
	for _, b := range list {
		if b.ID == a {
			return true
		}
	}
	return false
}

func TestGetOnCalls(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeClient := new(backendfakes.FakeClient)
	fakeLogger := new(loggingfakes.FakeLogger)
	client := New(fakeClient, fakeLogger)

	// Test error with OnCalls
	fakeClient.ListOnCallsStub = func(escalationPolicyId string) (*backend.ListOnCallsResponse, error) {
		return nil, errors.New("This is an error")
	}

	resp, err := client.GetOncallUsers("ignored", 1)

	a.Nil(resp)
	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "Failed to get Oncall Users:"), "error message should start with Failed to get Oncall Users: - got `%v`", err.Error())

	// Test return empty result
	fakeClient.ListOnCallsStub = func(escalationPolicyId string) (*backend.ListOnCallsResponse, error) {
		return &backend.ListOnCallsResponse{}, nil
	}

	resp, err = client.GetOncallUsers("ignored", 1)

	a.Nil(resp)
	a.NotNil(err)
	a.EqualValues("No Oncall Users returned", err.Error())

	// Set up valid responses
	fakeClient.ListOnCallsStub = func(escalationPolicyId string) (*backend.ListOnCallsResponse, error) {
		return onCalls, nil
	}

	// Test No phone numbers Found (escalation level 3 only has users with email specified, not phone number)
	resp, err = client.GetOncallUsers("ignored", 3)

	a.EqualValues(0, len(resp))
	a.Nil(err)

	// Test successful Oncalls request
	resp, err = client.GetOncallUsers("ignored", 1)
	a.Nil(err)
	a.NotNil(resp)

	a.EqualValues(2, len(resp))
	a.True(userInList("USER1", resp))
	a.True(userInList("USER2", resp))
}

func TestGetTeamUsers(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeClient := new(backendfakes.FakeClient)
	fakeLogger := new(loggingfakes.FakeLogger)
	client := New(fakeClient, fakeLogger)

	// Test erroring PD call
	fakeClient.ListTeamUsersStub = func(teamID string) (*backend.ListUsersResponse, error) {
		return nil, errors.New("Failed")
	}

	var maxTeamUsersAge = 0 * time.Minute

	users := client.GetTeamUsers("teamID", nil, maxTeamUsersAge)
	a.EqualValues(0, len(users))

	// Test null list
	fakeClient.ListTeamUsersStub = func(teamID string) (*backend.ListUsersResponse, error) {
		return &backend.ListUsersResponse{}, nil
	}

	users = client.GetTeamUsers("teamID", nil, maxTeamUsersAge)
	a.EqualValues(0, len(users))

	// Test successful request
	fakeClient.ListTeamUsersStub = func(teamID string) (*backend.ListUsersResponse, error) {
		return &backend.ListUsersResponse{
			Users: []*backend.User{
				&backend.User{
					ID: "USER1",
				},
				&backend.User{
					ID: "USER2",
				},
			},
		}, nil
	}

	// Test successful request
	users = client.GetTeamUsers("teamID", nil, maxTeamUsersAge)
	a.EqualValues(2, len(users))

	// Test successful filtered request
	users = client.GetTeamUsers("teamID", []string{"USER2"}, maxTeamUsersAge)
	a.EqualValues(1, len(users))
}
