package ssmparameterresolver

import (
	"code.justin.tv/event-engineering/goldengate/pkg/aws/backend/backendfakes"
	loggingfakes "code.justin.tv/event-engineering/goldengate/pkg/logging/backend/backendfakes"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
	"testing"
)

var paramMap = map[string]string{
	"/goldengate/twilioAccountSid":       "TWILIO_ACCOUNT_SID",
	"/goldengate/twilioAuthToken":        "TWILIO_AUTH_TOKEN",
	"/goldengate/pagerDutyApiKey":        "PD_API_KEY",
	"/goldengate/pdTwilioIntegrationKey": "PD_TWILIO_INTEGRATION_KEY",
	"/goldengate/jiraToken":              "JIRA_TOKEN",
	"/goldengate/jiraTokenSecret":        "JIRA_TOKEN_SECRET",
	"/goldengate/jiraPrivateKey":         "JIRA_PRIVATE_KEY",
}

func TestGetSSMParametersSSMError(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeLogger := new(loggingfakes.FakeLogger)
	fakeAWSBackend := new(backendfakes.FakeClient)

	r := New(fakeLogger, fakeAWSBackend)

	// Test erroring call
	fakeAWSBackend.SSMGetParametersStub = func(input *ssm.GetParametersInput) (*ssm.GetParametersOutput, error) {
		return nil, errors.New("SSM error")
	}

	_, err := r.GetParameters()
	a.NotNil(err)
	a.EqualValues("SSM error", err.Error())
}

func TestGetSSMParametersSuccess(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeLogger := new(loggingfakes.FakeLogger)
	fakeAWSBackend := new(backendfakes.FakeClient)

	r := New(fakeLogger, fakeAWSBackend)

	// Test successful call
	fakeAWSBackend.SSMGetParametersStub = func(input *ssm.GetParametersInput) (*ssm.GetParametersOutput, error) {
		output := &ssm.GetParametersOutput{
			Parameters: make([]*ssm.Parameter, 0),
		}
		for k, v := range paramMap {
			key := k
			value := v
			paramType := ssm.ParameterTypeSecureString
			output.Parameters = append(output.Parameters, &ssm.Parameter{Name: &key, Value: &value, Type: &paramType})
		}

		return output, nil
	}

	params, err := r.GetParameters()
	a.Nil(err)
	a.EqualValues(1, fakeAWSBackend.SSMGetParametersCallCount())
	a.EqualValues("TWILIO_ACCOUNT_SID", params.TwilioAccountSid)
	a.EqualValues("TWILIO_AUTH_TOKEN", params.TwilioAuthToken)
	a.EqualValues("PD_API_KEY", params.PagerDutyAPIKey)
	a.EqualValues("PD_TWILIO_INTEGRATION_KEY", params.PagerdutyTwilioIntegrationKey)
	a.EqualValues("JIRA_TOKEN", params.JiraToken)
	a.EqualValues("JIRA_TOKEN_SECRET", params.JiraTokenSecret)
	a.EqualValues("JIRA_PRIVATE_KEY", params.JiraPrivateKey)
}
