package backend

import (
	"context"
	goTwilio "github.com/kevinburke/twilio-go"
	"io"
	"net/http"
	"net/url"
	"time"
)

// Client redefines goTwilio.Client to allow custom methods and easier data mocking for tests.
// If this interface changes, counterfeiter must be re-run using `make mocks`
type Client interface {
	Call(from string, to string, u *url.URL) (*goTwilio.Call, error)
	Hangup(sid string) (*goTwilio.Call, error)
	GetExpectedSignature(host string, authToken string, URL string, postForm url.Values) (expectedTwilioSignature string)
	GetConference(ctx context.Context, sid string) (*goTwilio.Conference, error)
	GetConferences(start time.Time, end time.Time, data url.Values) goTwilio.ConferencePageIterator
	GetCall(ctx context.Context, sid string) (*goTwilio.Call, error)
	RetrieveRecording(recordingURL string) (io.ReadCloser, error)
	GetConferenceParticipants(ctx context.Context, conferenceSid string) (*ConferenceParticipantsResponse, error)
	GetConferenceRecordings(ctx context.Context, conferenceSid string) (*ConferenceRecordingsResponse, error)
}

type ConferenceParticipantsResponse struct {
	Participants []*goTwilio.Participant `json:"participants"`
}

type ConferenceRecordingsResponse struct {
	Recordings []*goTwilio.Recording `json:"recordings"`
}

type client struct {
	httpClient *http.Client
	twilio     *goTwilio.Client
}

// New creates a new Twilio client
func New(accountSid, authToken string, httpClient *http.Client) Client {
	return &client{
		httpClient: httpClient,
		twilio:     goTwilio.NewClient(accountSid, authToken, httpClient),
	}
}

func (c *client) Call(from string, to string, u *url.URL) (*goTwilio.Call, error) {
	return c.twilio.Calls.MakeCall(from, to, u)
}

func (c *client) Hangup(sid string) (*goTwilio.Call, error) {
	return c.twilio.Calls.Hangup(sid)
}

func (c *client) GetExpectedSignature(host string, authToken string, URL string, postForm url.Values) (expectedTwilioSignature string) {
	return goTwilio.GetExpectedTwilioSignature(host, authToken, URL, postForm)
}

func (c *client) GetConference(ctx context.Context, sid string) (*goTwilio.Conference, error) {
	return c.twilio.Conferences.Get(ctx, sid)
}

func (c *client) GetConferences(start time.Time, end time.Time, data url.Values) goTwilio.ConferencePageIterator {
	return c.twilio.Conferences.GetConferencesInRange(start, end, data)
}

func (c *client) GetConferenceParticipants(ctx context.Context, conferenceSid string) (*ConferenceParticipantsResponse, error) {
	// This is a bit hacky because the underlying SDK doesn't have functions for this call :(
	resp := &ConferenceParticipantsResponse{}
	err := c.twilio.GetResource(ctx, "Conferences", conferenceSid+"/Participants", resp)
	return resp, err
}

func (c *client) GetConferenceRecordings(ctx context.Context, conferenceSid string) (*ConferenceRecordingsResponse, error) {
	// This is a bit hacky because the underlying SDK doesn't have functions for this call :(
	resp := &ConferenceRecordingsResponse{}
	err := c.twilio.GetResource(ctx, "Conferences", conferenceSid+"/Recordings", resp)
	return resp, err
}

func (c *client) GetCall(ctx context.Context, sid string) (*goTwilio.Call, error) {
	return c.twilio.Calls.Get(ctx, sid)
}

func (c *client) RetrieveRecording(recordingURL string) (io.ReadCloser, error) {
	resp, err := c.httpClient.Get(recordingURL)

	if err != nil {
		return nil, err
	}

	return resp.Body, nil
}
