package twilio

import (
	logging "code.justin.tv/event-engineering/goldengate/pkg/logging/backend"
	"code.justin.tv/event-engineering/goldengate/pkg/twilio/backend"
	"context"
	"fmt"
	goTwilio "github.com/kevinburke/twilio-go"
	"github.com/pkg/errors"
	"io"
	"net/url"
)

// Client is a Twilio client containing methods for use in the goldengate system
type Client struct {
	twilio backend.Client
	logger logging.Logger
}

// New creates a new twilio client with functions for use in goldengate
func New(client backend.Client, logger logging.Logger) *Client {
	return &Client{
		twilio: client,
		logger: logger,
	}
}

// DialNumber calls the supplied number with the supplied pickup URL
func (c *Client) DialNumber(from, to, onPickupURL string) (string, error) {
	url, err := url.Parse(onPickupURL)
	if err != nil {
		return "", errors.Wrap(err, "Failed to parse pickup URL")
	}
	call, err := c.twilio.Call(from, to, url)
	if err != nil {
		return "", errors.Wrap(err, "Failed to dial number")
	}

	c.logger.Debugf("Created Call [%v] to %v", call.Sid, to)

	return call.Sid, nil
}

// Hangup  hangs up the call with the supplied sid
func (c *Client) Hangup(sid string) (*goTwilio.Call, error) {
	return c.twilio.Hangup(sid)
}

// ValidateRequest tests the supplied signature against a signature calculated using the auth token, returning an error if the signature does not match
func (c *Client) ValidateRequest(signature, host string, authToken string, URL string, postForm url.Values) error {
	expectedSig := c.twilio.GetExpectedSignature(host, authToken, URL, postForm)
	if expectedSig != signature {
		return fmt.Errorf("Invalid signature: %v - %v%v - %v", signature, host, URL, postForm.Encode())
	}
	return nil
}

// IsConferenceOpen will return true if there is a conference with the supplied name in progress
func (c *Client) IsConferenceOpen(ctx context.Context, conferenceName string) bool {
	filters := make(url.Values)
	filters.Set("Status", "in-progress")
	filters.Set("FriendlyName", conferenceName)

	conferences := c.twilio.GetConferences(goTwilio.Epoch, goTwilio.HeatDeath, filters)

	for page, err := conferences.Next(ctx); err == nil; {
		if len(page.Conferences) > 0 {
			c.logger.Debugf("Found conference with SID %v - %v - %v", page.Conferences[0].Sid, page.Conferences[0].Status, page.Conferences[0].FriendlyName)
			return true
		}
	}

	return false
}

// GetConference will return a conference object for the supplied CID
func (c *Client) GetConference(ctx context.Context, sid string) (*goTwilio.Conference, error) {
	return c.twilio.GetConference(ctx, sid)
}

// GetConferenceByFriendlyName will return a conference object for the supplied friendlyName
func (c *Client) GetConferenceByFriendlyName(ctx context.Context, friendlyName string) (*goTwilio.Conference, error) {
	filters := make(url.Values)
	filters.Set("Status", "in-progress")
	filters.Set("FriendlyName", friendlyName)

	conferences := c.twilio.GetConferences(goTwilio.Epoch, goTwilio.HeatDeath, filters)

	for page, err := conferences.Next(ctx); err == nil; {
		if len(page.Conferences) > 0 {
			return page.Conferences[0], nil
		}
	}

	return nil, errors.New("No conference found")
}

// GetConferenceParticipants will return the participant list for the supplied conference
func (c *Client) GetConferenceParticipants(ctx context.Context, conferenceSid string) (*backend.ConferenceParticipantsResponse, error) {
	return c.twilio.GetConferenceParticipants(ctx, conferenceSid)
}

// GetConferenceRecordings will return the recordings list for the supplied conference
func (c *Client) GetConferenceRecordings(ctx context.Context, conferenceSid string) (*backend.ConferenceRecordingsResponse, error) {
	return c.twilio.GetConferenceRecordings(ctx, conferenceSid)
}

// GetCall will return a call object for the supplied SID
func (c *Client) GetCall(ctx context.Context, sid string) (*goTwilio.Call, error) {
	return c.twilio.GetCall(ctx, sid)
}

// RetrieveRecording will return a ReadCloser for the supplied recording URL
func (c *Client) RetrieveRecording(recordingURL string) (io.ReadCloser, error) {
	return c.twilio.RetrieveRecording(recordingURL)
}
