package twilio

import (
	"code.justin.tv/event-engineering/goldengate/pkg/twilio/backend"
	"code.justin.tv/event-engineering/goldengate/pkg/twilio/backend/backendfakes"
	goTwilio "github.com/kevinburke/twilio-go"

	loggingfakes "code.justin.tv/event-engineering/goldengate/pkg/logging/backend/backendfakes"
	"context"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
	"io"
	"net/url"
	"strings"
	"testing"
	"time"
)

func TestSignatureValidation(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeClient := new(backendfakes.FakeClient)
	fakeLogger := new(loggingfakes.FakeLogger)
	client := New(fakeClient, fakeLogger)

	// Use the real GetExpectedSignature from the base twilio package to test things
	fakeClient.GetExpectedSignatureStub = goTwilio.GetExpectedTwilioSignature

	values, err := url.ParseQuery("From=+445556789&To=+155567234")
	a.Nil(err)

	// Test valid signature
	err = client.ValidateRequest("a1q8briwzLLwdxY7L5uU0+8J78M=", "www.example.com", "this_is_a_fake_auth_token", "/some_url", values)
	a.Nil(err)

	// Test invalid signature
	err = client.ValidateRequest("invalid_sig", "www.example.com", "this_is_a_fake_auth_token", "/some_url", values)
	a.NotNil(err)
}

type conferenceIterator struct{}

func (i *conferenceIterator) Next(context.Context) (*goTwilio.ConferencePage, error) {
	return &goTwilio.ConferencePage{
		Conferences: []*goTwilio.Conference{
			&goTwilio.Conference{
				FriendlyName: "hello",
			},
		},
	}, nil
}

type emptyConferenceIterator struct{}

func (i *emptyConferenceIterator) Next(context.Context) (*goTwilio.ConferencePage, error) {
	return nil, goTwilio.NoMoreResults
}

func TestGetConferences(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeClient := new(backendfakes.FakeClient)
	fakeLogger := new(loggingfakes.FakeLogger)
	client := New(fakeClient, fakeLogger)

	ctx := context.TODO()

	// Test with in progress conference
	fakeClient.GetConferencesStub = func(start time.Time, end time.Time, data url.Values) goTwilio.ConferencePageIterator {
		return &conferenceIterator{}
	}

	isOpen := client.IsConferenceOpen(ctx, "hello")

	a.True(isOpen)

	// Test by friendly name
	conference, err := client.GetConferenceByFriendlyName(ctx, "my_conference")
	a.Nil(err)
	a.EqualValues("hello", conference.FriendlyName)

	// Test with no in progress conferences
	fakeClient.GetConferencesStub = func(start time.Time, end time.Time, data url.Values) goTwilio.ConferencePageIterator {
		return &emptyConferenceIterator{}
	}

	isOpen = client.IsConferenceOpen(ctx, "hello")

	a.False(isOpen)

	// Test by friendly name
	conference, err = client.GetConferenceByFriendlyName(ctx, "my_conference")
	a.Nil(conference)
	a.NotNil(err)
	a.EqualValues("No conference found", err.Error())

	// Test Get Conference by Sid
	fakeClient.GetConferenceStub = func(ctx context.Context, sid string) (*goTwilio.Conference, error) {
		a.EqualValues("conference_sid", sid)

		return &goTwilio.Conference{
			Status: "conf_status",
		}, nil
	}

	conference, err = client.GetConference(ctx, "conference_sid")
	a.Nil(err)
	a.EqualValues("conf_status", conference.Status)
	a.EqualValues(1, fakeClient.GetConferenceCallCount())

}

func TestDialNumber(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeClient := new(backendfakes.FakeClient)
	fakeLogger := new(loggingfakes.FakeLogger)
	client := New(fakeClient, fakeLogger)

	// Test with in progress conference
	fakeClient.CallStub = func(from string, to string, u *url.URL) (*goTwilio.Call, error) {
		return &goTwilio.Call{
			Sid: "ABC123",
		}, nil
	}

	// Test invalid pickup URL
	_, err := client.DialNumber("+1234567890", "+1987654321", "\n:")

	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "Failed to parse pickup URL:"), "error message should start with Failed to parse pickup URL: - got `%v`", err.Error())

	// Test valid call
	_, err = client.DialNumber("+1234567890", "+1987654321", "https://example.com")

	a.Nil(err)

	// Test failed calls
	fakeClient.CallStub = func(from string, to string, u *url.URL) (*goTwilio.Call, error) {
		return nil, errors.New("Invalid number")
	}

	_, err = client.DialNumber("+1234567890", "+1987654321", "https://example.com")

	a.NotNil(err)
	a.True(strings.HasPrefix(err.Error(), "Failed to dial number:"), "error message should start with Failed to dial number: - got `%v`", err.Error())
}

func TestGetCall(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeClient := new(backendfakes.FakeClient)
	fakeLogger := new(loggingfakes.FakeLogger)
	client := New(fakeClient, fakeLogger)

	// Test get call
	fakeClient.GetCallStub = func(ctx context.Context, sid string) (*goTwilio.Call, error) {
		a.EqualValues("ABC123", sid)
		return &goTwilio.Call{
			Sid: "ABC123",
		}, nil
	}

	call, err := client.GetCall(context.TODO(), "ABC123")

	a.Nil(err)
	a.EqualValues("ABC123", call.Sid)
	a.EqualValues(1, fakeClient.GetCallCallCount())
}

func TestHangup(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeClient := new(backendfakes.FakeClient)
	fakeLogger := new(loggingfakes.FakeLogger)
	client := New(fakeClient, fakeLogger)

	// Test hangup
	fakeClient.HangupStub = func(sid string) (*goTwilio.Call, error) {
		a.EqualValues("ABC123", sid)
		return &goTwilio.Call{
			Sid: "ABC123",
		}, nil
	}

	call, err := client.Hangup("ABC123")

	a.Nil(err)
	a.EqualValues("ABC123", call.Sid)
	a.EqualValues(1, fakeClient.HangupCallCount())
}

func TestRetrieveRecording(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeClient := new(backendfakes.FakeClient)
	fakeLogger := new(loggingfakes.FakeLogger)
	client := New(fakeClient, fakeLogger)

	// Test get recording
	fakeClient.RetrieveRecordingStub = func(url string) (io.ReadCloser, error) {
		a.EqualValues("this is a test", url)
		return nil, nil
	}

	body, err := client.RetrieveRecording("this is a test")

	a.Nil(body)
	a.Nil(err)
	a.EqualValues(1, fakeClient.RetrieveRecordingCallCount())
}

func TestGetConferenceParticipants(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeClient := new(backendfakes.FakeClient)
	fakeLogger := new(loggingfakes.FakeLogger)
	client := New(fakeClient, fakeLogger)

	// Test get conference participants
	fakeClient.GetConferenceParticipantsStub = func(ctx context.Context, sid string) (*backend.ConferenceParticipantsResponse, error) {
		a.EqualValues("conf_sid", sid)

		return &backend.ConferenceParticipantsResponse{
			Participants: []*goTwilio.Participant{
				&goTwilio.Participant{
					CallSid: "call_sid",
				},
			},
		}, nil
	}

	participants, err := client.GetConferenceParticipants(context.TODO(), "conf_sid")
	a.Nil(err)
	a.NotNil(participants)
	a.EqualValues(1, fakeClient.GetConferenceParticipantsCallCount())
	a.EqualValues(1, len(participants.Participants))
	a.EqualValues("call_sid", participants.Participants[0].CallSid)
}

func TestGetConferenceRecordings(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	fakeClient := new(backendfakes.FakeClient)
	fakeLogger := new(loggingfakes.FakeLogger)
	client := New(fakeClient, fakeLogger)

	// Test get conference participants
	fakeClient.GetConferenceRecordingsStub = func(ctx context.Context, sid string) (*backend.ConferenceRecordingsResponse, error) {
		a.EqualValues("conf_sid", sid)

		return &backend.ConferenceRecordingsResponse{
			Recordings: []*goTwilio.Recording{
				&goTwilio.Recording{
					Sid: "recording_sid",
				},
			},
		}, nil
	}

	recordings, err := client.GetConferenceRecordings(context.TODO(), "conf_sid")
	a.Nil(err)
	a.NotNil(recordings)
	a.EqualValues(1, fakeClient.GetConferenceRecordingsCallCount())
	a.EqualValues(1, len(recordings.Recordings))
	a.EqualValues("recording_sid", recordings.Recordings[0].Sid)
}
