package main

import (
	"flag"
	"os"
	"os/signal"
	"path/filepath"

	"code.justin.tv/event-engineering/goosechase/modules/bashrunner"
	"code.justin.tv/event-engineering/goosechase/modules/edgetest"
	"code.justin.tv/event-engineering/goosechase/modules/ingestspeed"
	"code.justin.tv/event-engineering/goosechase/modules/ip"
	"code.justin.tv/event-engineering/goosechase/modules/mtr"
	"code.justin.tv/event-engineering/goosechase/modules/pinger"
	"code.justin.tv/event-engineering/goosechase/modules/playercore"
	"code.justin.tv/event-engineering/goosechase/modules/speedtest"
	"code.justin.tv/event-engineering/goosechase/modules/videotiming"

	"code.justin.tv/event-engineering/goosechase/modules/location"
	"code.justin.tv/event-engineering/goosechase/pkg/collector"
	"code.justin.tv/event-engineering/goosechase/pkg/creds"
	"code.justin.tv/event-engineering/goosechase/pkg/finish"
	"code.justin.tv/event-engineering/goosechase/pkg/interfaces"
	"code.justin.tv/event-engineering/goosechase/pkg/tcpdump"
	"code.justin.tv/event-engineering/goosechase/pkg/testplaylist"
	"code.justin.tv/event-engineering/goosechase/pkg/testrunner"
	"code.justin.tv/event-engineering/goosechase/pkg/util"
	log "github.com/sirupsen/logrus"
	"time"
	"io"
)

// These are compiled in to the binary at build time
var AWSAccessKeyID string
var AWSSecretAccessKey string
var OutputS3Bucket string
var IngestStreamKey string
var GitVersion string
var BuildTime string

func main() {
	logger := log.StandardLogger()

	dir, err := filepath.Abs(filepath.Dir(os.Args[0]))
	if err != nil {
		log.Fatal(err)
	}

	var skipRoot = flag.Bool("noroot", false, "Skip the elevated check, will skip certain tests")
	var skipUpload = flag.Bool("skipupload", false, "Ignore uploading to S3")
	var comment = flag.String("comment", "", "Run comment")
	var ignoreDump = flag.Bool("ignoredump", false, "Skip the TCPDump")
	var playlist = flag.String("playlist", dir+"/testconf.json", "What test playlist to run")
	var gpsDeviceName = flag.String("gpsdevice", "/dev/tty.usbserial", "If you have a serial GPS device, what device is it?")
	var gpsDeviceBaudRate = flag.Uint("gpsdevicebaudrate", 4800, "If you have a serial GPS device, what baud rate is it?")
	var playercoreLocation = flag.String("playercore", dir+"/bin/streamplayer", "Configure alternate player-core binary location")
	var scriptLocation = flag.String("scriptpath", dir+"/scripts/", "Change the location of where scripts are located")
	var logpath = flag.String("logpath", "logs", "Change the location of where tests are saved")
	var debuglogpath = flag.String("debuglogpath", "debug_logs", "If set, will log to this file as well as stdout")
	var networkInterfaceOverride = flag.String("interface", "", "What network interface to packet capture. Autodetects on OSX")

	flag.Parse()

	var runElevated = !*skipRoot
	var testCompleted = false

	debugLog, err := os.OpenFile(*debuglogpath, os.O_APPEND|os.O_WRONLY|os.O_CREATE, 0766)
	if err != nil {
		panic(err)
	}
	defer debugLog.Close()

	mw := io.MultiWriter(os.Stdout, debugLog)
	logger.SetOutput(mw)
	log.SetOutput(mw)

	//Check if user has elevated privileges, needed for running certain tests.
	if runElevated && os.Geteuid() != 0 {
		log.Error("Please run goosechase as root")
		os.Exit(255)
	}
	if !runElevated {
		log.Warn("Goosechase is starting without elevated privileges. Some tests will be skipped.")
	}

	if *ignoreDump {
		log.Warn("TCPDump will be skipped")
	}

	if *skipUpload {
		log.Warn("Uploading to S3 disabled")
	}

	//Empty load default to creds.json
	gcCreds, err := creds.Load(dir + "/creds.json")
	if err != nil {
		log.Info("Could not load credentials, using build time credentials")
		gcCreds = &creds.Credentials{
			AWS: creds.AWS{
				AccessKey: AWSAccessKeyID,
				Secret:    AWSSecretAccessKey,
				Bucket:    OutputS3Bucket,
			},
			Ingest: creds.Ingest{
				StreamKey: IngestStreamKey,
			},
		}
	}

	//Get primary network interface
	var networkInterface string

	if *networkInterfaceOverride == "" {
		networkInterface, err = interfaces.GetPrimary()
		if err != nil {
			log.WithFields(log.Fields{
				"Error": err,
			}).Error("Could not detect primary network interace, using en0")
			networkInterface = "en0"
		}
	} else {
		networkInterface = *networkInterfaceOverride
		log.Infof("Using override interface from command line %s", networkInterface)
	}

	//Create a new testrunner instance and load modules
	tr := testrunner.New(runElevated)
	tr.RegisterModules([]testrunner.TestModule{
		ip.New(),
		videotiming.New(),
		speedtest.New(),
		pinger.New(),
		edgetest.New(),
		playercore.New(*playercoreLocation),
		mtr.New(logger),
		ingestspeed.New(dir, logger),
		bashrunner.New(*scriptLocation),
	})

	//Load the test playlist from file
	pl, err := testplaylist.Load(*playlist)
	if err != nil {
		log.WithFields(log.Fields{
			"Filename": *playlist,
		}).Panic("Could not load Playlist")
	}

	log.WithFields(log.Fields{
		"Elevated":           runElevated,
		"Registered Modules": tr.GetRegisteredModules(),
		"Loaded Playlist":    *playlist,
		"Tests to run":       len(pl.Playlist),
		"Git Version":        GitVersion,
		"Build Time":         BuildTime,
	}).Info("Started Goosechase")

	//Ask user for comment on test
	if *comment == "" {
		c := util.GetComment()
		comment = &c
	}

	log.Printf("Using comment: %v\n", *comment)

	locationProvider := location.New(dir+"/bin/corelocationcli", *gpsDeviceName, *gpsDeviceBaudRate, 30*time.Second)

	//Try to acquire location data.
	locationData := locationProvider.GetLocation()

	//Setup Collector
	col := collector.New(*logpath,
		*comment,
		collector.S3Config{
			Key:    gcCreds.AWS.AccessKey,
			Secret: gcCreds.AWS.Secret,
			Bucket: gcCreds.AWS.Bucket,
		},
		locationData,
		GitVersion,
		BuildTime,
	)

	//Save initial report to get location and time.
	col.SaveReport()

	//Handle CTRL+C
	c := make(chan os.Signal, 1)
	signal.Notify(c, os.Interrupt)
	go func() {
		for sig := range c {
			_ = sig
			if !testCompleted {
				log.WithFields(log.Fields{
					"Completed Tests":     col.CurrentReport.CompletedTests,
					"Failed Tests":        col.CurrentReport.FailedTests,
					"Number of tests ran": tr.Testcounter + 1,
					"Test Duration":       col.CurrentReport.TestDuration,
				}).Info("Aborting test, saving logs.")

				col.FinishReport(false)
			}

			os.Exit(0)
		}
	}()

	//Range over the playlist and run each tests
	for _, mc := range pl.Playlist {
		// Give the test a name, if a suffix is configured, add that to the test name.
		testname := mc.Module
		if mc.Suffix != "" {
			testname = testname + "-" + mc.Suffix
		}

		//Start TCPDump
		dump := tcpdump.New(networkInterface, col.Testpath+"/tcpdumps/"+testname, logger)
		if tr.ShouldDump(mc.Module) && runElevated && !*ignoreDump {
			err := dump.Run()
			if err != nil {
				log.WithFields(log.Fields{
					"Error": err,
				}).Warn("Could not start tcpdump")
			}
		}

		//Run the test and collect data.
		data, err := tr.Run(mc)
		if err != nil {
			log.Error(err)
			col.LogError(err)
			//If test fail, tell this to collector so it can mark test as failed.
			col.FailTest(testname)
		}

		//If TCPDump is running, stop it.
		if tr.ShouldDump(mc.Module) && runElevated && !*ignoreDump {
			dump.Stop()
		}

		//Save the test result.
		col.SaveTest(testname, data)

	}

	testCompleted = true

	//Finish report saves report.json and adds end timings
	col.FinishReport(true)

	//Upload all json files to S3.
	if !*skipUpload {
		col.Upload()
	}

	log.WithFields(log.Fields{
		"Completed Tests":     col.CurrentReport.CompletedTests,
		"Failed Tests":        col.CurrentReport.FailedTests,
		"Number of tests ran": tr.Testcounter + 1,
		"Test Duration":       col.CurrentReport.TestDuration,
	}).Info("👍👍👍GOOSECHASE IS DONE👍👍👍")

	log.Info("🔉 TIME FOR MUSIC 🔉")

	finish.Finish(dir + "/polar.mp3")
}
