package edgetest

import (
	"encoding/json"
	"io/ioutil"
	"net/http"
	"time"

	log "github.com/sirupsen/logrus"
)

type EdgeResult struct {
	Completed bool    `json:"Completed"`
	Error     error   `json:"Error"`
	Download  float64 `json:"Download"`
	Latency   Latency `json:"Latency"`
	PoP       string  `json:"PoP"`
}

type Latency struct {
	Completed    bool
	Error        error
	Measurements []time.Duration
	MinLatency   float64
	AvgLatency   float64
	MaxLatency   float64
}

type EdgeTest struct {
}

func New() *EdgeTest {
	return &EdgeTest{}
}

func (v *EdgeTest) Name() (appname string, printname string) {
	return "edgetest", "Twitch Video Edge Speedtest"
}

func (v *EdgeTest) Elevated() bool {
	return false
}

func (v *EdgeTest) TCPDump() bool {
	return true
}

func (v *EdgeTest) Run(config map[string]string) ([]byte, error) {

	edgeurl := config["loadfile"]
	latencyurl := config["latencyfile"]

	log.WithFields(log.Fields{
		"File": edgeurl,
	}).Info("Starting Twitch Video Edge Speedtest")

	l, err := GetLatency(latencyurl)
	if err != nil {
		//return nil, err
		l.Error = err
		l.Completed = false
		err = nil
	}

	start := time.Now()

	client := http.Client{
		Timeout: 30 * time.Second,
	}

	req, err := http.NewRequest("GET", edgeurl, nil)
	if err != nil {
		return nil, err
	}
	req.Header.Set("Cache-Control", "no-cache")

	resp, err := client.Do(req)
	if err != nil {
		return nil, err
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		if err.Error() == "net/http: request canceled (Client.Timeout exceeded while reading body)" {
			//We expect this error to happen.
			err = nil
		} else {
			return nil, err
		}
	}
	finish := time.Now()
	bodyLen := len(body)

	defer func() {
		err = resp.Body.Close()
		if err != nil {
			log.WithFields(log.Fields{
				"Error": err,
			}).Info("Error closing body of download request:")
		}
	}()

	cluster := resp.Header["X-Source-Cluster"]
	var pop string

	for _, c := range cluster {
		pop = pop + c
	}

	bits := float64(bodyLen * 8)
	megabits := bits / float64(1000) / float64(1000)
	seconds := finish.Sub(start).Seconds()
	mbps := megabits / seconds

	log.WithFields(log.Fields{
		"Download": mbps,
		"PoP":      pop,
	}).Info("Edge speedtest done")

	eg := EdgeResult{
		Download: mbps,
		PoP:      pop,
		Latency:  l,
	}

	b, err := json.MarshalIndent(eg, "", "	")
	if err != nil {
		return nil, err
	}

	return b, err
}

func GetLatency(url string) (Latency, error) {
	var latency time.Duration
	var minLatency time.Duration
	var maxLatency time.Duration
	var avgLatency time.Duration
	var measurements []time.Duration

	var NUMTESTS = 5

	for i := 0; i < NUMTESTS; i++ {
		var failed bool
		var finish time.Time

		start := time.Now()

		client := http.Client{
			Timeout: 10 * time.Second,
		}

		req, err := http.NewRequest("GET", url, nil)
		if err != nil {
			return Latency{}, err
		}

		req.Header.Set("Cache-Control", "no-cache")

		resp, err := client.Do(req)

		if err != nil {
			return Latency{}, err
		}

		defer func() {
			err = resp.Body.Close()
			if err != nil {
				log.Printf("error closing body of latency request: %v", err)
			}
		}()

		finish = time.Now()
		_, err = ioutil.ReadAll(resp.Body)
		if err != nil {
			return Latency{}, err
		}

		if failed {
			latency = 1 * time.Minute
		} else {
			latency = finish.Sub(start)
		}

		measurements = append(measurements, latency)

		if maxLatency == 0 {
			maxLatency = latency
		} else if latency > maxLatency {
			maxLatency = latency
		}

		if minLatency == 0 {
			minLatency = latency
		} else if latency < minLatency {
			minLatency = latency
		}
		avgLatency = avgLatency + latency

	}

	result := Latency{
		Completed:    true,
		Error:        nil,
		Measurements: measurements,
		MinLatency:   float64(time.Duration(minLatency.Nanoseconds())*time.Nanosecond) / 1000000,
		MaxLatency:   float64(time.Duration(maxLatency.Nanoseconds())*time.Nanosecond) / 1000000,
		AvgLatency:   float64(time.Duration(avgLatency.Nanoseconds())*time.Nanosecond) / 1000000 / float64(NUMTESTS),
	}

	return result, nil

}
