package ingestspeed

import (
	"context"
	"encoding/json"
	"fmt"
	"os"
	"strings"
	"time"

	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	rtmp "code.justin.tv/event-engineering/gortmp/pkg/rtmp"
	findingest "code.justin.tv/event-engineering/rtmp/pkg/findingest"
	flvlooper "code.justin.tv/event-engineering/rtmp/pkg/flvlooper"
)

type Result struct {
	IngestServer string
	Results      []ResultItem
}

type ResultItem struct {
	Timestamp time.Time
	Kbps      float64
}

type Ingest struct {
	logger  logging.Logger
	rootDir string
}

func New(rootDir string, l logging.Logger) *Ingest {
	return &Ingest{
		logger:  l,
		rootDir: rootDir,
	}
}

func (v *Ingest) Name() (appname string, printname string) {
	return "ingest", "Ingest Speedtest"
}

func (v *Ingest) Elevated() bool {
	return false
}

func (v *Ingest) TCPDump() bool {
	return true
}

func (v *Ingest) Run(config map[string]string) ([]byte, error) {

	flvFile := config["flvfile"]
	if flvFile == "" {
		flvFile = v.rootDir + "/bitrate_testing.flv"
	}

	var re *Result

	rtmpEndpoint := config["rtmp_endpoint"]
	if rtmpEndpoint != "" {
		items, rtmpErr := runStream(flvFile, config["rtmp_endpoint"], v.logger)
		if rtmpErr != nil {
			return nil, rtmpErr
		}

		re = &Result{
			IngestServer: rtmpEndpoint,
			Results:      items,
		}
	} else {
		var fiErr error
		re, fiErr = runFindIngest(flvFile, config["streamkey"], v.logger)
		if fiErr != nil {
			return nil, fiErr
		}
	}

	b, err := json.MarshalIndent(re, "", "	")
	if err != nil {
		return nil, err
	}

	return b, err
}

func runFindIngest(flvFile, streamKey string, logger logging.Logger) (*Result, error) {
	ingests, err := findingest.Find()
	if err != nil {
		return nil, fmt.Errorf("Failed to find best ingest %v", err)
	}

	if len(ingests) < 1 {
		return nil, fmt.Errorf("No results returned from find-ingest")
	}

	bestIngest := ingests[0]

	logger.Infof("Using %v as best ingest from find-ingest", bestIngest.URLTemplate)

	ingestURL := strings.Replace(bestIngest.URLTemplate, "{stream_key}", streamKey, 1) + "?bandwidthtest=true"

	items, err := runStream(flvFile, ingestURL, logger)

	if err != nil {
		return nil, err
	}

	return &Result{
		IngestServer: bestIngest.URLTemplate,
		Results:      items,
	}, nil
}

func runStream(flvFile, ingestPoint string, logger logging.Logger) ([]ResultItem, error) {
	fileR, fileErr := os.OpenFile(flvFile, os.O_RDONLY, 0600)

	if fileErr != nil {
		return nil, fmt.Errorf("Failed to open file for ingest stream test %v", fileErr)
	}

	looper, err := flvlooper.Create(fileR)

	if err != nil {
		return nil, fmt.Errorf("Failed to create FLV looper %v", err)
	}

	ms := rtmp.NewMediaStream(context.Background(), "output")

	errCh := make(chan error)

	// Create a new RTMP pusher
	pusher := NewPusher(errCh, logger)

	// Initiate the connection to the output URL and push the data from the supplied mediastream to the server
	err = pusher.Push(ingestPoint, ms)

	if err != nil {
		return nil, fmt.Errorf("Failed to create pusher %v", err)
	}

	logger.Infof("Pushing RTMP to %v", ingestPoint)

	result := run(looper, ms)

	return result, nil
}
