package ingestspeed

import (
	"code.justin.tv/event-engineering/gortmp/pkg/rtmp"
	"code.justin.tv/event-engineering/rtmp/pkg/flvlooper"
	"github.com/pkg/errors"
	"io"
	"log"
	"time"
)

func run(flvFile *flvlooper.File, outputStream rtmp.MediaStream) []ResultItem {
	defer flvFile.Close()

	var thisLoopStartTs uint32
	var fileStartTs uint32
	var lastTs uint32
	ref := time.Now()

	var bytesPushed int
	var lastClock time.Time
	var startClock = time.Now()
	var results = make([]ResultItem, 0)
	var maxKbps float64

	for {
		if time.Since(startClock).Minutes() > 5 {
			break
		}

		header, bytes, err := flvFile.ReadTag()

		if err != nil {
			// If err == io.EOF then we're at the end of the file, so finish
			if err == io.EOF {
				break
			} else {
				// We got an unexpected error, so we're going to break here
				log.Println(errors.Wrap(err, "Unexpected error from flvFile.ReadTag"))
				outputStream.Close()
				break
			}
		}

		tag := &rtmp.FlvTag{
			Type:      header.TagType,
			Timestamp: header.Timestamp,
			Size:      header.DataSize,
			Bytes:     bytes,
		}

		bytesPushed += len(bytes)

		if lastClock.IsZero() {
			lastClock = time.Now()
		}

		if time.Since(lastClock).Seconds() > 1 {
			clockDeltaMs := time.Since(lastClock).Seconds() * 1000
			Kbps := (float64(bytesPushed) / float64(clockDeltaMs)) * 1000 * 8 / 1024

			results = append(results, ResultItem{
				Timestamp: time.Now(),
				Kbps:      Kbps,
			})

			if Kbps > maxKbps {
				maxKbps = Kbps
				log.Printf("Max Bitrate: %f Kb/s", Kbps)
			}

			bytesPushed = 0
			lastClock = time.Now()
		}

		if tag.Timestamp == 0 {
			tag.Timestamp = lastTs
		} else {
			if fileStartTs == 0 {
				fileStartTs = tag.Timestamp
			}

			// Make timestamps relative to 0
			tag.Timestamp = tag.Timestamp - fileStartTs

			// Work out how long we want to pause for, otherwise we just shove things in buffers as fast as we can and things get clogged up
			clock := time.Since(ref)
			target := time.Millisecond * time.Duration(tag.Timestamp)
			waitTime := target - clock

			if waitTime > 500*time.Millisecond {
				// clock drift of >500ms means we need to adjust our reference
				log.Printf("adjusting for clock drift of %s", waitTime)
				ref = ref.Add(-waitTime)
			} else if waitTime > 0 {
				time.Sleep(waitTime)
			}

			// Fix timestamps as we're looping
			tag.Timestamp = thisLoopStartTs + tag.Timestamp

			// I'm not sure why this happens sometimes, but I had a source FLV where the last tag in the file had a timestamp that was lower than the previous one
			// Just coding around it for now until I can find out more about if this is valid or I should rework the source media
			if tag.Timestamp < lastTs {
				tag.Timestamp = lastTs
			}

			lastTs = tag.Timestamp

			// Check if it's closed first
			if outputStream.IsClosed() {
				log.Println("Outputstream closed, ending loop")
				break
			}
		}

		// Push tag to output stream
		err = outputStream.Publish(tag)
		if err != nil {
			log.Println(err)
		}
	}

	return results
}
