package ip

import (
	"encoding/json"
	"io/ioutil"
	"net"
	"net/http"
	"strings"

	log "github.com/sirupsen/logrus"
)

type IPResult struct {
	Local string
	ExtV4 string
	ExtV6 string
}

type IP struct {
}

func New() *IP {
	return &IP{}
}

func (v *IP) Name() (appname string, printname string) {
	return "ip", "IP Collector"
}

func (v *IP) Elevated() bool {
	return false
}

func (v *IP) TCPDump() bool {
	return false
}

func (v *IP) Run(config map[string]string) ([]byte, error) {

	localIP, err := GetLocalIP()
	if err != nil {
		log.WithFields(log.Fields{
			"Error": err,
		}).Warn("Could not get local IP")
	}
	ipv4, v4err := GetExternalIPv4()
	if v4err != nil {
		log.WithFields(log.Fields{
			"Error": v4err,
		}).Warn("Could not get IPv4 addr")
	}
	ipv6, v6err := GetExternalIPv6()
	if v6err != nil {
		log.WithFields(log.Fields{
			"Error": v6err,
		}).Warn("Could not get IPv6 addr")
	}

	log.WithFields(log.Fields{
		"Local IP": localIP.String(),
		"IPv4":     ipv4,
		"IPv6":     ipv6,
	}).Info("Found IP addresses")

	ip := IPResult{
		Local: localIP.String(),
		ExtV4: ipv4,
		ExtV6: ipv6,
	}

	b, err := json.MarshalIndent(ip, "", "	")
	if err != nil {
		return nil, err
	}

	return b, nil
}

func GetLocalIP() (net.IP, error) {
	conn, err := net.Dial("udp", "8.8.8.8:53")
	if err != nil {
		log.Fatal(err)
		return nil, nil
	}
	defer conn.Close()

	localAddr := conn.LocalAddr().(*net.UDPAddr)

	return localAddr.IP, nil
}

func GetExternalIPv4() (string, error) {
	resp, err := http.Get("http://ipv4.icanhazip.com/")
	if err != nil {
		return "", err
	}
	defer resp.Body.Close()

	body, httperr := ioutil.ReadAll(resp.Body)
	if httperr != nil {
		return "", err
	}

	resIP := strings.TrimSuffix(string(body), "\n")

	return resIP, nil
}

func GetExternalIPv6() (string, error) {
	resp, err := http.Get("http://ipv6.icanhazip.com/")
	if err != nil {
		return "", err
	}
	defer resp.Body.Close()

	body, httperr := ioutil.ReadAll(resp.Body)
	if httperr != nil {
		return "", err
	}

	resIP := strings.TrimSuffix(string(body), "\n")

	return resIP, nil
}
