package location

import (
	"code.justin.tv/event-engineering/goosechase/pkg/corelocation"
	"code.justin.tv/event-engineering/goosechase/pkg/gps"
	"runtime"
	"strconv"
	"sync"
	"time"

	log "github.com/sirupsen/logrus"
)

// How accurate does a GPS fix need to be in order for it to be acceptable https://en.wikipedia.org/wiki/Dilution_of_precision_(navigation)
const hdopThreshold = 5

type LocationData struct {
	Completed bool        `json:"completed"`
	Latitude  float64     `json:"latitude"`
	Longitude float64     `json:"longitude"`
	Metadata  interface{} `json:"metadata"`
}

type Provider interface {
	GetLocation() LocationData
}

type locationProvider struct {
	coreLocationCLI   string
	gpsDeviceName     string
	gpsDeviceBaudRate uint
	gpsMaxWaitTime    time.Duration
}

func New(coreLocationCLI, gpsDeviceName string, gpsDeviceBaudRate uint, gpsMaxWaitTime time.Duration) Provider {
	return &locationProvider{
		coreLocationCLI:   coreLocationCLI,
		gpsDeviceName:     gpsDeviceName,
		gpsDeviceBaudRate: gpsDeviceBaudRate,
		gpsMaxWaitTime:    gpsMaxWaitTime,
	}
}

func (p *locationProvider) GetLocation() LocationData {
	// Attempt to use the CoreLocation API if we're on mac
	if runtime.GOOS == "darwin" {
		log.Info("Attempting to get location via CoreLocation")
		resp, err := corelocation.Get(p.coreLocationCLI)
		if err != nil {
			log.Infof("Could not get location via CoreLocation %v", err)
		} else {
			completed := true
			fltLat, err := strconv.ParseFloat(resp.Latitude, 64)
			if err != nil {
				completed = false
				log.Infof("Could not parse latitude from value %v", resp.Latitude)
			}
			fltLon, err := strconv.ParseFloat(resp.Longitude, 64)
			if err != nil {
				completed = false
				log.Infof("Could not parse longitude from value %v", resp.Longitude)
			}

			return LocationData{
				Completed: completed,
				Latitude:  fltLat,
				Longitude: fltLon,
				Metadata:  resp,
			}
		}
	}

	if p.gpsDeviceName != "" {
		log.Info("Attempting to get location via GPS device")
		gpsClient := gps.New(p.gpsDeviceName, p.gpsDeviceBaudRate)

		ch, err := gpsClient.Start()
		if err != nil {
			log.Infof("Could not initialise GPS client %v", err)
		} else {
			var wg sync.WaitGroup
			wg.Add(1)

			ld := &LocationData{
				Completed: false,
			}

			go func(locationData *LocationData) {
				defer wg.Done()
				start := time.Now()

				for {
					if time.Since(start) >= p.gpsMaxWaitTime {
						return
					}

					if pos, ok := <-ch; ok {
						locationData.Latitude = pos.Latitude
						locationData.Longitude = pos.Longitude
						locationData.Metadata = pos

						if pos.HDOP != 0 && pos.HDOP <= hdopThreshold {
							locationData.Completed = true
							return
						}
					} else {
						return
					}
				}
			}(ld)

			wg.Wait()

			gpsClient.Stop()

			return *ld
		}
	}

	return LocationData{
		Completed: false,
		Metadata:  "No location providers could be found",
	}
}
