package mtr

import (
	"encoding/json"
	"regexp"
	"time"

	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	"code.justin.tv/event-engineering/goosechase/pkg/twitchutil"
	"github.com/tonobo/mtr/pkg/hop"
	"github.com/tonobo/mtr/pkg/mtr"
)

type Result struct {
	Address string
	Hops    map[int]*hop.HopStatistic
}

var extractHostRe = regexp.MustCompile("^https://([^/]+).*$")

type MTR struct {
	logger logging.Logger
}

func New(l logging.Logger) *MTR {
	return &MTR{
		logger: l,
	}
}

func (v *MTR) Name() (appname string, printname string) {
	return "mtr", "MTR"
}

func (v *MTR) Elevated() bool {
	return true
}

func (v *MTR) TCPDump() bool {
	return true
}

func (v *MTR) Run(config map[string]string) ([]byte, error) {

	MTRAddress := config["mtraddress"]

	if MTRAddress == "" {

		reqToken, _, err := twitchutil.GetToken("streamerhouse")
		if err != nil {
			return nil, err
		}
		masterManifest, _, err := twitchutil.GetMasterManifest("streamerhouse", reqToken)
		if err != nil {
			return nil, err
		}

		mpl, err := twitchutil.ParseMasterManifest(masterManifest)
		if err != nil {
			return nil, err
		}

		chunked := twitchutil.ExtractVariant(mpl.Variants, "chunked")

		MTRAddress = extractHostRe.ReplaceAllString(chunked.URI, "$1")
	}

	v.logger.Infof("Performing MTR to %v", MTRAddress)

	mtrClient, ch, err := mtr.NewMTR(MTRAddress, "", time.Millisecond*800, time.Millisecond*1000, time.Nanosecond, 64, 10, 50, true)

	if err != nil {
		return nil, err
	}

	go func(ch chan struct{}) {
		for {
			if _, ok := <-ch; !ok {
				return
			}
		}
	}(ch)

	mtrClient.Run(ch, 30)
	close(ch)

	r := Result{
		Address: mtrClient.SrcAddress,
		Hops:    mtrClient.Statistic,
	}

	b, err := json.MarshalIndent(r, "", "	")
	if err != nil {
		return nil, err
	}

	return b, nil
}
