package pinger

import (
	"encoding/json"
	"net"
	"time"

	log "github.com/sirupsen/logrus"
	"github.com/sparrc/go-ping"
)

type Statistics struct {
	// PacketsRecv is the number of packets received.
	PacketsRecv int

	// PacketsSent is the number of packets sent.
	PacketsSent int

	// PacketLoss is the percentage of packets lost.
	PacketLoss float64

	// IPAddr is the address of the host being pinged.
	IPAddr *net.IPAddr

	// Addr is the string address of the host being pinged.
	Addr string

	// Rtts is all of the round-trip times sent via this pinger.
	Rtts []time.Duration

	// MinRtt is the minimum round-trip time sent via this pinger.
	MinRtt time.Duration

	// MaxRtt is the maximum round-trip time sent via this pinger.
	MaxRtt time.Duration

	// AvgRtt is the average round-trip time sent via this pinger.
	AvgRtt time.Duration

	// StdDevRtt is the standard deviation of the round-trip times sent via
	// this pinger.
	StdDevRtt time.Duration
}

type Pinger struct {
}

func New() *Pinger {
	return &Pinger{}
}

func (v *Pinger) Name() (appname string, printname string) {
	return "ping", "Ping"
}

func (v *Pinger) Elevated() bool {
	//return false
	return true
}

func (v *Pinger) TCPDump() bool {
	return true
}

func (v *Pinger) Run(config map[string]string) ([]byte, error) {

	var Result []Statistics

	var destinations []string

	for _, v := range config {
		destinations = append(destinations, v)
	}

	for _, des := range destinations {

		pinger, err := ping.NewPinger(des)
		if err != nil {
			return nil, err
		}
		pinger.Count = 5
		pinger.Timeout = 10 * time.Second
        pinger.SetPrivileged(true)
		pinger.Run()
		stats := pinger.Statistics()

		log.WithFields(log.Fields{
			"Avg RTT": stats.AvgRtt,
			"Max RTT": stats.MaxRtt,
			"Min RTT": stats.MinRtt,
		}).Info("Pinged " + des)

		np := Statistics{
			PacketsRecv: stats.PacketsRecv,
			PacketsSent: stats.PacketsSent,
			PacketLoss:  stats.PacketLoss,
			IPAddr:      stats.IPAddr,
			Addr:        stats.Addr,
			Rtts:        stats.Rtts,
			MinRtt:      stats.MinRtt,
			MaxRtt:      stats.MaxRtt,
			AvgRtt:      stats.AvgRtt,
			StdDevRtt:   stats.StdDevRtt,
		}

		Result = append(Result, np)
	}

	b, err := json.MarshalIndent(Result, "", "	")
	if err != nil {
		return nil, err
	}

	return b, nil
}
