package playercore

import (
	"bufio"
	"encoding/json"
	"os/exec"
	"regexp"
	"strconv"
	"syscall"
	"time"

	log "github.com/sirupsen/logrus"
)

type PlayerMetrics struct {
	MasterManifestReady   []MasterManifestReady
	VideoPlay             []VideoPlay
	ABSStreamFormatChange []ABSStreamFormatChange
	NSecondPlay           []NSecondPlay
	BufferEmpty           []BufferEmpty
	Buffer2Seconds        []Buffer2Seconds
	MinuteWatched         []MinuteWatched
	BufferRefill          []BufferRefill
	VideoError            []VideoError
	PlayerErrors          []string
}

type PlayerCore struct {
	binaryLocation string
}

func New(binaryLocation string) *PlayerCore {
	return &PlayerCore{
		binaryLocation: binaryLocation,
	}
}

func (v *PlayerCore) Name() (appname string, printname string) {
	return "playercore", "Player Core"
}

func (v *PlayerCore) Elevated() bool {
	return false
}

func (v *PlayerCore) TCPDump() bool {
	return true
}

func (v *PlayerCore) Run(config map[string]string) ([]byte, error) {

	channel := config["channel"]
	lowLatency := config["lowlatency"]
	abr := config["abr"]
	dur := config["duration"]

	//SANITY
	if channel == "" {
		channel = "streamerhouse"
	}

	if channel == "" {
		lowLatency = "true"
	}

	if channel == "" {
		abr = "true"
	}

	if dur == "" {
		channel = "60"
	}

	duration, err := strconv.Atoi(dur)
	if err != nil {
		return nil, err
	}

	var metrics PlayerMetrics

	var filter = regexp.MustCompile("^([^{]+)(.*)$")
	var playererror = regexp.MustCompile(`^Player error\s(.*)$`)

	cmd := exec.Command(v.binaryLocation, "https://twitch.tv/"+channel, lowLatency, abr)

	//Kill player if it takes to long
	timer := time.NewTimer(time.Second * time.Duration(duration))
	go func(timer *time.Timer, cmd *exec.Cmd) {
		for _ = range timer.C {
			cmd.Process.Signal(syscall.SIGINT)
		}
	}(timer, cmd)

	dl, _ := strconv.ParseBool(lowLatency)
	al, _ := strconv.ParseBool(abr)

	log.WithFields(log.Fields{
		"Channel":     channel,
		"Duration":    duration,
		"Low Latency": dl,
		"ABR":         al,
	}).Info("Launching player-core binary")

	stdout, err := cmd.StdoutPipe()
	if err != nil {
		return nil, err
	}

	stderr, err := cmd.StderrPipe()
	if err != nil {
		return nil, err
	}

	stdin, err := cmd.StdinPipe()
	if err != nil {
		return nil, err
	}
	defer stdin.Close()

	if err := cmd.Start(); err != nil {
		return nil, err
	}

	errin := bufio.NewScanner(stderr)
	go func() {
		for errin.Scan() {
			ex := playererror.FindStringSubmatch(errin.Text())
			if len(ex) > 0 {
				if ex[1] != "signal: interrupt" {
					metrics.PlayerErrors = append(metrics.PlayerErrors, ex[1])
					log.WithFields(log.Fields{
						"Error": ex[1],
					}).Error("Player-core threw error")
					if ex[1] == "stopping playback" {
						log.Error("Player-core gave up and stopped playback. Module exits.")
						cmd.Process.Signal(syscall.SIGINT)
					}
				}
			}
			//	fmt.Println(errin.Text())
		}
	}()

	in := bufio.NewScanner(stdout)
	go func() {
		for in.Scan() {
			eventtype := filter.FindStringSubmatch(in.Text())[1]
			data := filter.FindStringSubmatch(in.Text())[2]
			switch eventtype {
			case "master_manifest_ready":
				d, err := parseMasterManifestReady(data)
				if err != nil {
					log.Error(err)
				}
				metrics.MasterManifestReady = append(metrics.MasterManifestReady, d)
				log.WithFields(log.Fields{
					"TimeToMasterPlaylistReady":   d.TimeToMasterPlaylistReady,
					"TimeToMasterPlaylistRequest": d.TimeToMasterPlaylistRequest,
				}).Info("Got Master Manifest Ready event")
			case "video-play":
				d, err := parseVideoPlay(data)
				if err != nil {
					log.Error(err)
				}
				metrics.VideoPlay = append(metrics.VideoPlay, d)
				log.WithFields(log.Fields{
					"TimeToSegmentReady":   d.TimeToSegmentReady,
					"TimeToSegmentRequest": d.TimeToSegmentRequest,
					"VideoBufferSize":      d.VideoBufferSize,
				}).Info("Got Video Play event")
			case "abs_stream_format_change":
				d, err := parseABSStreamFormatChange(data)
				if err != nil {
					log.Error(err)
				}
				metrics.ABSStreamFormatChange = append(metrics.ABSStreamFormatChange, d)
				log.WithFields(log.Fields{
					"CurrentStreamFormatBitrate": d.CurrentStreamFormatBitrate,
					"CurrentBitrate":             d.CurrentBitrate,
					"VideoBufferSize":            d.VideoBufferSize,
				}).Info("Got ABS Stream Format Change event")
			case "n_second_play":
				d, err := parseNSecondPlay(data)
				if err != nil {
					log.Error(err)
				}
				metrics.NSecondPlay = append(metrics.NSecondPlay, d)
				log.WithFields(log.Fields{
					"PlaySessionID":   d.PlaySessionID,
					"CurrentBitrate":  d.CurrentBitrate,
					"VideoBufferSize": d.VideoBufferSize,
				}).Info("Got NSecondPlay event")
			case "buffer-empty":
				d, err := parseBufferEmpty(data)
				if err != nil {
					log.Error(err)
				}
				metrics.BufferEmpty = append(metrics.BufferEmpty, d)
				log.WithFields(log.Fields{
					"BufferEmptyCount": d.BufferEmptyCount,
					"CurrentBitrate":   d.CurrentBitrate,
					"VideoBufferSize":  d.VideoBufferSize,
				}).Info("Got Buffer Empty event")
			case "buffer_2_seconds":
				d, err := parseBuffer2Seconds(data)
				if err != nil {
					log.Error(err)
				}
				metrics.Buffer2Seconds = append(metrics.Buffer2Seconds, d)
				log.WithFields(log.Fields{
					"BufferEmptyCount": d.BufferEmptyCount,
					"CurrentBitrate":   d.CurrentBitrate,
					"VideoBufferSize":  d.VideoBufferSize,
				}).Info("Got Buffer 2 Seconds Event")
			case "minute-watched":
				d, err := parseMinuteWatched(data)
				if err != nil {
					log.Error(err)
				}
				metrics.MinuteWatched = append(metrics.MinuteWatched, d)
				log.WithFields(log.Fields{
					"AverageBitrate":  d.AverageBitrate,
					"CurrentBitrate":  d.CurrentBitrate,
					"VideoBufferSize": d.VideoBufferSize,
				}).Info("Got Minute Watched event")
			case "buffer-refill":
				d, err := parseBufferRefill(data)
				if err != nil {
					log.Error(err)
				}
				metrics.BufferRefill = append(metrics.BufferRefill, d)
				log.WithFields(log.Fields{
					"CurrentBitrate":  d.CurrentBitrate,
					"VideoBufferSize": d.VideoBufferSize,
				}).Info("Got Buffer Refill event")
			case "video_error":
				d, err := parseVideoError(data)
				if err != nil {
					log.Error(err)
				}
				metrics.VideoError = append(metrics.VideoError, d)
				log.WithFields(log.Fields{
					"Error Code":        d.VideoErrorCode,
					"Error Recoverable": d.VideoErrorRecoverable,
					"Channel":           channel,
				}).Info("Got Video Error event")
			default:
				log.WithFields(log.Fields{
					"Type": eventtype,
					"Data": data,
				}).Warn("Got Unimplemented Event: " + eventtype)
			}

		}
	}()

	if err := cmd.Wait(); err != nil {
		if err.Error() != "signal: interrupt" {
			m, err := prepareMetrics(metrics)
			if err != nil {
				return nil, err
			}
			return m, err
		}
	}

	m, err := prepareMetrics(metrics)
	if err != nil {
		return nil, err
	}
	return m, err
}

func prepareMetrics(pm PlayerMetrics) ([]byte, error) {
	b, err := json.MarshalIndent(pm, "", "	")
	if err != nil {
		return nil, err
	}
	return b, err
}
