package speedtest

import (
	"encoding/json"
	"fmt"

	"github.com/kylegrantlucas/speedtest"
	log "github.com/sirupsen/logrus"
)

type SpeedtestResult struct {
	Download        float64 `json:"Download"`
	Upload          float64 `json:"Upload"`
	LatencyMeasured bool    `json:"LatencyMeasured"`
	Latency         float64 `json:"Latency"`
	URL             string  `json:"URL"`
	Lat             float64 `json:"Latitude"`
	Lon             float64 `json:"Longitude"`
	Country         string  `json:"Country"`
}

type SpeedTest struct {
}

func New() *SpeedTest {
	return &SpeedTest{}
}

func (v *SpeedTest) Name() (appname string, printname string) {
	return "speedtest", "Speedtest"
}

func (v *SpeedTest) Elevated() bool {
	return false
}

func (v *SpeedTest) TCPDump() bool {
	return true
}

func (v *SpeedTest) Run(config map[string]string) ([]byte, error) {
	client, err := speedtest.NewDefaultClient()
	if err != nil {
		return nil, fmt.Errorf("error creating client: %v", err)
	}

	var latencyMeasure = true

	server, err := client.GetServer(config["server"])
	if err != nil {
		if server.URL == "" {
			return nil, fmt.Errorf("error getting server: %v", err)
		}

		latencyMeasure = false

		err = nil
	}

	log.WithFields(log.Fields{
		"Server Name": server.Name + " - " + server.Sponsor,
		"Country":     server.Country,
		"Lat":         server.Lat,
		"Lon":         server.Lon,
	}).Info("Starting Speedtest")

	log.Info("Speedtesting Download")

	dmbps, err := client.Download(server)
	if err != nil {
		fmt.Println(dmbps)
		return nil, fmt.Errorf("error getting download: %v", err)
	}
	log.Info("Speedtesting Upload")
	umbps, err := client.Upload(server)
	if err != nil {
		return nil, fmt.Errorf("error getting upload: %v", err)
	}

	sp := SpeedtestResult{
		Download:        dmbps,
		Upload:          umbps,
		LatencyMeasured: latencyMeasure,
		Latency:         server.Latency,
		URL:             server.URL,
		Lat:             server.Lat,
		Lon:             server.Lon,
		Country:         server.Country,
	}

	log.WithFields(log.Fields{
		"Download": dmbps,
		"Upload":   umbps,
		"Latency":  server.Latency,
	}).Info("Speedtest Completed")

	b, err := json.MarshalIndent(sp, "", "	")
	if err != nil {
		return nil, err
	}

	return b, nil
}
