package videotiming

import (
	"encoding/json"
	"io"
	"net/http"
	"strings"
	"time"

	log "github.com/sirupsen/logrus"

	"code.justin.tv/event-engineering/goosechase/pkg/twitchutil"
	"github.com/grafov/m3u8"
)

type Timing struct {
	Channel          string
	Token            float64
	MasterManifest   float64
	Playlist         float64
	FirstSegmentByte float64
}

type VideoTiming struct {
}

func New() *VideoTiming {
	return &VideoTiming{}
}

func (v *VideoTiming) Name() (appname string, printname string) {
	return "videotiming", "Video Timings"
}

func (v *VideoTiming) Elevated() bool {
	return false
}

func (v *VideoTiming) TCPDump() bool {
	return true
}

/////// TESTS VIDEO TIMINGS
func (v *VideoTiming) Run(config map[string]string) ([]byte, error) {
	channel := config["channel"]
	log.WithFields(log.Fields{
		"Channel": channel,
	}).Info("Starting Video Timing Test")

	reqToken, tokenTime, err := twitchutil.GetToken(channel)
	if err != nil {
		return nil, err
	}

	masterManifest, mmTime, err := twitchutil.GetMasterManifest(channel, reqToken)
	if err != nil {
		return nil, err
	}

	playlist, plTime, err := twitchutil.GetWeaverVariant(masterManifest, "chunked")

	parsedPl, err := parsePlaylist(playlist)
	if err != nil {
		return nil, err
	}

	var segmenturl string

	for i, s := range parsedPl.Segments {
		if i > 1 {
			segmenturl = s.URI
			break
		}
	}

	timings := Timing{
		Channel:        channel,
		Token:          float64(tokenTime) / 1000000,
		MasterManifest: float64(mmTime) / 1000000,
		Playlist:       float64(plTime) / 1000000,
	}

	if segmenturl != "" {
		st, err := timeToFirstByte(segmenturl)
		if err != nil {

		} else {
			timings.FirstSegmentByte = float64(st) / 1000000
		}
	}

	log.WithFields(log.Fields{
		"Time to auth Token":         timings.Token,
		"Time to Master Manifest":    timings.MasterManifest,
		"Time to Playlist":           timings.Playlist,
		"time to First Segment Byte": timings.FirstSegmentByte,
	}).Info("Video Timing test for " + channel + " completed")

	b, err := json.MarshalIndent(timings, "", "	")
	if err != nil {
		return nil, err
	}

	return b, nil
}

func timeToFirstByte(segmentURL string) (time.Duration, error) {

	segmentClient := http.Client{
		Timeout: time.Second * 20,
	}

	req, err := http.NewRequest(http.MethodGet, segmentURL, nil)
	if err != nil {
		return 0, err
	}

	start := time.Now()

	res, err := segmentClient.Do(req)
	if err != nil {
		return 0, err
	}

	buf := make([]byte, 1)
	for {
		n, err := res.Body.Read(buf)
		if err == io.EOF {
			break
		}
		if n > 0 {
			break
		}
	}

	finish := time.Now()

	requestTime := finish.Sub(start)

	return requestTime, nil
}

func parsePlaylist(playlist string) (m3u8.MediaPlaylist, error) {
	b := strings.NewReader(playlist)
	p, listType, err := m3u8.DecodeFrom(b, true)
	if err != nil {
		panic(err)
	}
	switch listType {
	case m3u8.MEDIA:
		mediapl := p.(*m3u8.MediaPlaylist)
		return *mediapl, nil
	}
	return m3u8.MediaPlaylist{}, nil
}
