package collector

import (
	"bytes"
	"encoding/binary"
	"encoding/json"
	"io/ioutil"
	"os"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	log "github.com/sirupsen/logrus"

	"code.justin.tv/event-engineering/goosechase/modules/location"
)

type Collector struct {
	key        string
	secret     string
	Bucket     string
	Logpath    string
	Testpath   string
	Testfolder string

	CurrentReport Report

	//coral is super lazy
	s3data map[string][]byte
}

type S3Config struct {
	Key    string
	Secret string
	Bucket string
}

type Report struct {
	Comment        string                `json:"Comment"`
	CompletedTests []string              `json:"CompletedTests"`
	FailedTests    []string              `json:"FailedTests"`
	Location       location.LocationData `json:"Location"`
	StartTime      time.Time             `json:"StartTime"`
	FinishTime     time.Time             `json:"FinishTime"`
	TestDuration   time.Duration         `json:"TestDuration"`
	Errors         []error               `json:"Errors"`
	Completed      bool                  `json:"Completed"`
	GitVersion     string                `json:"GitVersion"`
	BuildTime      string                `json:"BuildTime"`
}

func New(logpath string, comment string, s3 S3Config, lc location.LocationData, gitversion string, buildtime string) *Collector {

	loc, _ := time.LoadLocation("UTC")
	now := time.Now().In(loc)

	testfolder := now.Format(time.RFC3339)

	os.MkdirAll(logpath+"/"+testfolder, 0777)
	os.MkdirAll(logpath+"/"+testfolder+"/tcpdumps", 0777)

	c := Report{
		Comment:    comment,
		Location:   lc,
		GitVersion: gitversion,
		BuildTime:  buildtime,
		StartTime:  time.Now(),
	}

	return &Collector{
		CurrentReport: c,
		key:           s3.Key,
		secret:        s3.Secret,
		Bucket:        s3.Bucket,
		Logpath:       logpath,
		Testpath:      logpath + "/" + testfolder,
		Testfolder:    testfolder,
		s3data:        make(map[string][]byte),
	}
}

func (c *Collector) SaveReport() {

	b, err := json.MarshalIndent(c.CurrentReport, "", "	")
	if err != nil {
		log.Error("Could not marshal report. (WTF?)")
	}

	c.s3data["report"] = b

	err = ioutil.WriteFile(c.Testpath+"/report.json", b, 0777)
	if err != nil {
		log.Error("Could not write report to file.")
	}

}

func (c *Collector) FinishReport(completed bool) {

	c.CurrentReport.FinishTime = time.Now()
	c.CurrentReport.TestDuration = c.CurrentReport.FinishTime.Sub(c.CurrentReport.StartTime)
	c.CurrentReport.Completed = completed
	c.SaveReport()

}

func (c *Collector) SaveTest(testname string, json []byte) {

	c.CurrentReport.CompletedTests = append(c.CurrentReport.CompletedTests, testname)

	err := ioutil.WriteFile(c.Testpath+"/"+testname+".json", json, 0777)
	if err != nil {
		log.Error("Could not write report to file for test " + testname)
	}

	c.s3data[testname] = json

	c.SaveReport()

}

func (c *Collector) LogError(e error) {
	c.CurrentReport.Errors = append(c.CurrentReport.Errors, e)
}
func (c *Collector) FailTest(testname string) {
	c.CurrentReport.FailedTests = append(c.CurrentReport.FailedTests, testname)

	c.SaveReport()
}

func (c *Collector) Upload() {

	s, err := session.NewSession(
		&aws.Config{
			Region:      aws.String("us-west-2"),
			Credentials: credentials.NewStaticCredentials(c.key, c.secret, ""),
		})
	if err != nil {
		log.Error(err)
	}

	for k, d := range c.s3data {

		slicesize := int64(binary.Size(d))

		_, err = s3.New(s).PutObject(&s3.PutObjectInput{
			Bucket:               aws.String(c.Bucket),
			Key:                  aws.String(c.Testfolder + "/" + k + ".json"),
			ACL:                  aws.String("private"),
			Body:                 bytes.NewReader(d),
			ContentLength:        aws.Int64(slicesize),
			ContentType:          aws.String("application/json"),
			ContentDisposition:   aws.String("attachment"),
			ServerSideEncryption: aws.String("AES256"),
		})

		if err != nil {
			log.Error(err)
		}

		log.Info("Uploaded file to S3 as " + c.Testfolder + "/" + k + ".json")

	}

}
