package gps

import (
	"bufio"
	"github.com/adrianmo/go-nmea"
	"github.com/jacobsa/go-serial/serial"
	"io"
	"log"
	//"strings"
)

type Position struct {
	Latitude  float64 `json:"latitude"`
	Longitude float64 `json:"longitude"`
	HDOP      float64 `json:"hdop"`
}

type Client interface {
	Start() (chan Position, error)
	Stop()
}

type gps struct {
	deviceName string
	baudRate   uint
	port       io.ReadWriteCloser
	stopped    bool
}

func New(deviceName string, baudRate uint) Client {
	return &gps{
		deviceName: deviceName,
		baudRate:   baudRate,
	}
}

func (g *gps) Start() (chan Position, error) {
	g.stopped = false

	// Set up options.
	options := serial.OpenOptions{
		PortName:        g.deviceName,
		BaudRate:        g.baudRate,
		DataBits:        8,
		StopBits:        1,
		MinimumReadSize: 4,
	}

	port, err := serial.Open(options)
	if err != nil {
		return nil, err
	}

	g.port = port

	ch := g.readShit()

	return ch, nil
}

func (g *gps) readShit() chan Position {
	lineReader := bufio.NewScanner(g.port)
	ch := make(chan Position)

	go func() {
		for lineReader.Scan() {
			sentence := lineReader.Text()

			s, err := nmea.Parse(sentence)
			if err != nil {
				log.Printf("Failed to parse GPS sentence %v", sentence)
				log.Print(err)
				continue
			}

			if s.Prefix() == nmea.PrefixGPGGA {
				gga := s.(nmea.GPGGA)
				ch <- Position{
					gga.Latitude,
					gga.Longitude,
					gga.HDOP,
				}
			}
		}
		if err := lineReader.Err(); err != nil {
			if !g.stopped {
				log.Printf("Error reading from GPS device %v", err)
			}
			return
		}
	}()

	return ch
}

func (g *gps) Stop() {
	g.stopped = true
	g.port.Close()
}
