package interfaces

import (
	"bufio"
	"fmt"
	"os/exec"
	"regexp"
	"runtime"
	"strings"

	log "github.com/sirupsen/logrus"
)

func GetPrimary() (string, error) {
	var deviceLine *regexp.Regexp
	var out []byte
	var err error

	if runtime.GOOS == "darwin" {
		// Example:
		// a483e79e15b2:goosechase andrewf$ route get usher.ttvnw.net
		//    route to: 192.108.239.254
		// destination: default
		//        mask: default
		//     gateway: mymodem.modem
		//   interface: en0
		//       flags: <UP,GATEWAY,DONE,STATIC,PRCLONING>
		//  recvpipe  sendpipe  ssthresh  rtt,msec    rttvar  hopcount      mtu     expire
		//        0         0         0         0         0         0      1500         0
		deviceLine = regexp.MustCompile(`^\s*interface:\s(.*)$`)
		out, err = exec.Command("route", "get", "usher.ttvnw.net").Output()
	} else if runtime.GOOS == "linux" {
		// Example:
		// creep% route
		// Kernel IP routing table
		// Destination     Gateway         Genmask         Flags Metric Ref    Use Iface
		// default         218.100.43.113  0.0.0.0         UG    100    0        0 eth0
		// 218.100.43.112  *               255.255.255.248 U     0      0        0 eth0
		deviceLine = regexp.MustCompile(`^default\s+.*\s([a-z][^\s]+)$`)
		out, err = exec.Command("route").Output()
	} else {
		return "", fmt.Errorf("Interface detection is only supported on OSX and Linux")
	}

	if err != nil {
		return "", err
	}

	scanner := bufio.NewScanner(strings.NewReader(string(out)))
	for scanner.Scan() {
		line := scanner.Text()
		ex := deviceLine.FindStringSubmatch(line)
		if len(ex) > 0 {
			log.Info("Detected primary network interface: " + ex[1])
			return ex[1], err
		}
	}

	return "", fmt.Errorf("Could not parse an interface out of the route command")
}
