// +build !linux

package tcpdump

import (
	"fmt"
	"os"
	"sync"

	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	"github.com/google/gopacket"
	"github.com/google/gopacket/layers"
	"github.com/google/gopacket/pcap"
	"github.com/google/gopacket/pcapgo"
)

type tcpdump struct {
	networkInterface string
	outputFile       string
	stopped          bool
	logger           logging.Logger
	wg               sync.WaitGroup
}

func New(networkInterface, outputFile string, logger logging.Logger) TcpDump {
	return &tcpdump{
		networkInterface: networkInterface,
		outputFile:       outputFile,
		logger:           logger,
	}
}

func (t *tcpdump) Run() error {
	t.wg.Add(1)
	f, err := os.Create(fmt.Sprintf("%v.pcapng", t.outputFile))
	if err != nil {
		return err
	}

	r, err := pcapgo.NewNgWriter(f, layers.LinkTypeEthernet)
	if err != nil {
		return err
	}

	go func() {
		defer r.Flush()
		defer f.Close()
		defer t.wg.Done()

		if handle, err := pcap.OpenLive(t.networkInterface, 1600, true, pcap.BlockForever); err != nil {
			t.logger.Errorf("Error opening interface %v", err)
			return
		} else {
			packetSource := gopacket.NewPacketSource(handle, handle.LinkType())
			for packet := range packetSource.Packets() {
				packet.Metadata().CaptureInfo.InterfaceIndex = 0
				err = r.WritePacket(packet.Metadata().CaptureInfo, packet.Data())
				if err != nil {
					t.logger.Errorf("Error writing packet %v", err)
					return
				}

				if t.stopped {
					handle.Close()
				}
			}
		}
	}()

	return nil
}

func (t *tcpdump) Stop() {
	t.stopped = true
	t.wg.Wait()
}
