package testrunner

import (
	"fmt"
	"strconv"

	"code.justin.tv/event-engineering/goosechase/pkg/testplaylist"
	log "github.com/sirupsen/logrus"
)

type TestRunner struct {
	RegisteredModules map[string]TestModule
	Testcounter       int
	runelevated       bool
}

type TestModule interface {
	Run(config map[string]string) ([]byte, error)
	Name() (appname string, printname string)
	Elevated() bool
	TCPDump() bool
}

func New(runelevated bool) *TestRunner {
	return &TestRunner{
		RegisteredModules: make(map[string]TestModule),
		runelevated:       runelevated,
	}
}

func (t *TestRunner) RegisterModules(modules []TestModule) {
	for _, m := range modules {
		ap, _ := m.Name()
		t.RegisteredModules[ap] = m
	}
}

func (t *TestRunner) GetRegisteredModules() []string {
	var rm []string
	for _, m := range t.RegisteredModules {
		name, _ := m.Name()
		rm = append(rm, name)
	}
	return rm
}

func (t *TestRunner) ShouldDump(module string) bool {
	return t.RegisteredModules[module].TCPDump()
}

func (t *TestRunner) Run(mc testplaylist.ModuleConfig) ([]byte, error) {
	if t.RegisteredModules[mc.Module] == nil {
		return nil, fmt.Errorf("Cannot find module " + mc.Module + ". Check the config. Skipping this test")
	}
	_, modulename := t.RegisteredModules[mc.Module].Name()

	if t.RegisteredModules[mc.Module].Elevated() && !t.runelevated {
		return nil, fmt.Errorf(modulename + " requries elevated privileges, skipping test.")
	}

	log.WithFields(log.Fields{
		"Test #":      strconv.Itoa(t.Testcounter + 1),
		"Description": mc.Description,
	}).Info("Starting " + modulename + " module.")

	d, e := t.RegisteredModules[mc.Module].Run(mc.Config)
	if e != nil {
		log.WithFields(log.Fields{
			"Error": e,
		}).Error(modulename + " failed.")
	}

	t.Testcounter = t.Testcounter + 1

	return d, e
}
