package twitchutil

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"strings"
	"time"

	"github.com/grafov/m3u8"
)

type Token struct {
	Token            string    `json:"token"`
	Sig              string    `json:"sig"`
	MobileRestricted bool      `json:"mobile_restricted"`
	ExpiresAt        time.Time `json:"expires_at"`
}

func GetToken(channel string) (Token, time.Duration, error) {

	var rqurl = "https://api.twitch.tv/api/channels/" + channel + "/access_token?client_id=7xowrgquc8495rq4w5p79aeiwzxglu8&platform=VideoTesting"
	tokenClient := http.Client{
		Timeout: time.Second * 20,
	}

	req, err := http.NewRequest(http.MethodGet, rqurl, nil)
	if err != nil {
		return Token{}, 0, err
	}

	start := time.Now()

	res, err := tokenClient.Do(req)
	if err != nil {
		return Token{}, 0, err
	}

	body, err := ioutil.ReadAll(res.Body)
	if err != nil {
		return Token{}, 0, err
	}

	finish := time.Now()

	requestTime := finish.Sub(start)

	reqToken := Token{}
	err = json.Unmarshal(body, &reqToken)
	if err != nil {
		return Token{}, 0, err
	}

	return reqToken, requestTime, nil

}

func GetMasterManifest(channel string, token Token) (string, time.Duration, error) {

	params := url.Values{}
	params.Add("allow_source", "true")
	params.Add("fast_bread", "true")
	params.Add("token", token.Token)
	params.Add("sig", token.Sig)

	var rqurl = "https://usher.ttvnw.net/api/channel/hls/" + channel + ".m3u8?" + params.Encode()
	tokenClient := http.Client{
		Timeout: time.Second * 20,
	}

	req, err := http.NewRequest(http.MethodGet, rqurl, nil)
	if err != nil {
		return "", 0, err
	}

	start := time.Now()

	res, err := tokenClient.Do(req)
	if err != nil {
		return "", 0, err
	}

	body, err := ioutil.ReadAll(res.Body)
	if err != nil {
		return "", 0, err
	}

	finish := time.Now()

	requestTime := finish.Sub(start)

	return string(body), requestTime, nil

}

func GetWeaverVariant(mm string, requestedVariant string) (string, time.Duration, error) {

	mpl, err := ParseMasterManifest(mm)
	if err != nil {
		return "", 0, err
	}
	variant := ExtractVariant(mpl.Variants, "chunked")

	var rqurl = variant.URI
	tokenClient := http.Client{
		Timeout: time.Second * 20,
	}

	req, err := http.NewRequest(http.MethodGet, rqurl, nil)
	if err != nil {
		return "", 0, err
	}

	start := time.Now()

	res, err := tokenClient.Do(req)
	if err != nil {
		return "", 0, err
	}

	body, err := ioutil.ReadAll(res.Body)
	if err != nil {
		return "", 0, err
	}

	finish := time.Now()

	requestTime := finish.Sub(start)

	return string(body), requestTime, nil

}

func ExtractVariant(v []*m3u8.Variant, name string) m3u8.Variant {

	for _, v := range v {
		if v.Video == name {
			return *v
		}
	}
	return *v[0]

}

func ParseMasterManifest(mm string) (m3u8.MasterPlaylist, error) {
	b := strings.NewReader(mm)
	p, listType, err := m3u8.DecodeFrom(b, true)
	if err != nil {
		return m3u8.MasterPlaylist{}, err
	}
	switch listType {
	//	case m3u8.MEDIA:
	//mediapl := p.(*m3u8.MediaPlaylist)
	case m3u8.MASTER:
		masterpl := p.(*m3u8.MasterPlaylist)
		return *masterpl, nil
	}

	return m3u8.MasterPlaylist{}, fmt.Errorf("could not parse")
}
