package codec

import (
	"encoding/binary"
	"errors"
	"io"
)

type ByteStream interface {
	io.Reader
	io.ByteReader
}

// AVCDecoderConfigurationRecord from ISO/IEC 14496-15
type AVCDecoderConfig struct {
	ConfigurationVersion byte
	AvcProfileIndication byte
	ProfileCompatibility byte
	AvcLevelIndication   byte
	LengthSizeMinusOne   byte
	SpsTable             [][]byte
	PpsTable             [][]byte
}

var ErrUnknownVersion = errors.New("Unknown Version")

func readParamSet(r ByteStream) ([]byte, error) {
	var length uint16
	err := binary.Read(r, binary.BigEndian, &length)
	if err != nil {
		return nil, err
	}
	s := make([]byte, length)
	n, err := r.Read(s)
	if err != nil {
		return nil, err
	}
	if n != len(s) {
		return nil, io.EOF
	}
	return s, nil
}

func NewAVCDecoderConfig(r ByteStream) (*AVCDecoderConfig, error) {
	extra := &AVCDecoderConfig{}
	var err error

	extra.ConfigurationVersion, err = r.ReadByte()
	if err != nil {
		return nil, err
	}
	if extra.ConfigurationVersion != 1 {
		return nil, ErrUnknownVersion
	}

	var p [5]byte
	n, err := r.Read(p[:])
	if n != 5 {
		return nil, io.EOF
	}
	extra.AvcProfileIndication = p[0]
	extra.ProfileCompatibility = p[1]
	extra.AvcLevelIndication = p[2]
	extra.LengthSizeMinusOne = p[3] & 0x3
	numSPS := p[4] & 0x1F

	extra.SpsTable = make([][]byte, numSPS)
	for i := byte(0); i < numSPS; i++ {
		extra.SpsTable[i], err = readParamSet(r)
		if err != nil {
			return nil, err
		}
	}

	numPPS, err := r.ReadByte()
	if err != nil {
		return nil, err
	}

	extra.PpsTable = make([][]byte, numPPS)
	for i := byte(0); i < numPPS; i++ {
		extra.PpsTable[i], err = readParamSet(r)
		if err != nil {
			return nil, err
		}
	}

	return extra, nil
}
