package rtmp

import (
	"bytes"
	"encoding/binary"
	"fmt"
)

const (
	STREAM_BEGIN_EVENT       = 0
	STREAM_EOF_EVENT         = 1
	STREAM_DRY_EVENT         = 2
	SET_BUFFER_LENGTH_EVENT  = 3
	STREAM_IS_RECORDED_EVENT = 4
	PING_REQUEST_EVENT       = 5
	PING_RESPONSE_EVENT      = 6
)

type Event interface {
	RawMessage() (*RawMessage, error)
}

type StreamBeginEvent struct {
	StreamID uint32
}

func (e StreamBeginEvent) RawMessage() (*RawMessage, error) {
	buf := make([]byte, 6)
	binary.BigEndian.PutUint16(buf, STREAM_BEGIN_EVENT)
	binary.BigEndian.PutUint32(buf[2:], e.StreamID)

	return &RawMessage{
		ChunkStreamID: CS_ID_USER_CONTROL,
		Type:          USER_CONTROL_MESSAGE,
		Data:          bytes.NewBuffer(buf),
	}, nil
}

type StreamEOFEvent struct {
	StreamID uint32
}

func (e StreamEOFEvent) RawMessage() (*RawMessage, error) {
	buf := make([]byte, 6)
	binary.BigEndian.PutUint16(buf, STREAM_EOF_EVENT)
	binary.BigEndian.PutUint32(buf[2:], e.StreamID)

	return &RawMessage{
		ChunkStreamID: CS_ID_USER_CONTROL,
		Type:          USER_CONTROL_MESSAGE,
		Data:          bytes.NewBuffer(buf),
	}, nil
}

type StreamDryEvent struct {
	StreamID uint32
}

func (e StreamDryEvent) RawMessage() (*RawMessage, error) {
	buf := make([]byte, 6)
	binary.BigEndian.PutUint16(buf, STREAM_DRY_EVENT)
	binary.BigEndian.PutUint32(buf[2:], e.StreamID)

	return &RawMessage{
		ChunkStreamID: CS_ID_USER_CONTROL,
		Type:          USER_CONTROL_MESSAGE,
		Data:          bytes.NewBuffer(buf),
	}, nil
}

type SetBufferLengthEvent struct {
	StreamID     uint32
	BufferLength uint32
}

func (e SetBufferLengthEvent) RawMessage() (*RawMessage, error) {
	buf := make([]byte, 10)
	binary.BigEndian.PutUint16(buf, SET_BUFFER_LENGTH_EVENT)
	binary.BigEndian.PutUint32(buf[2:], e.StreamID)
	binary.BigEndian.PutUint32(buf[6:], e.BufferLength)

	return &RawMessage{
		ChunkStreamID: CS_ID_USER_CONTROL,
		Type:          USER_CONTROL_MESSAGE,
		Data:          bytes.NewBuffer(buf),
	}, nil
}

type StreamIsRecordedEvent struct {
	StreamID uint32
}

func (e StreamIsRecordedEvent) RawMessage() (*RawMessage, error) {
	buf := make([]byte, 10)
	binary.BigEndian.PutUint16(buf, STREAM_IS_RECORDED_EVENT)
	binary.BigEndian.PutUint32(buf[2:], e.StreamID)

	return &RawMessage{
		ChunkStreamID: CS_ID_USER_CONTROL,
		Type:          USER_CONTROL_MESSAGE,
		Data:          bytes.NewBuffer(buf),
	}, nil
}

type PingRequestEvent struct {
	Timestamp uint32
}

func (e PingRequestEvent) RawMessage() (*RawMessage, error) {
	buf := make([]byte, 6)
	binary.BigEndian.PutUint16(buf, PING_REQUEST_EVENT)
	binary.BigEndian.PutUint32(buf[2:], e.Timestamp)

	return &RawMessage{
		ChunkStreamID: CS_ID_USER_CONTROL,
		Type:          USER_CONTROL_MESSAGE,
		Data:          bytes.NewBuffer(buf),
	}, nil
}

type PingResponseEvent struct {
	Timestamp uint32
}

func (e PingResponseEvent) RawMessage() (*RawMessage, error) {
	buf := make([]byte, 6)
	binary.BigEndian.PutUint16(buf, PING_RESPONSE_EVENT)
	binary.BigEndian.PutUint32(buf[2:], e.Timestamp)

	return &RawMessage{
		ChunkStreamID: CS_ID_USER_CONTROL,
		Type:          USER_CONTROL_MESSAGE,
		Data:          bytes.NewBuffer(buf),
	}, nil
}

func ParseEvent(msg *RawMessage) (Event, error) {
	if msg.Type != USER_CONTROL_MESSAGE {
		return nil, fmt.Errorf("Invalid message type passed to ParseEvent: %d", msg.Type)
	}

	buf := bytes.NewReader(msg.Data.Bytes())
	var eventType uint16
	err := binary.Read(buf, binary.BigEndian, &eventType)
	if err != nil {
		return nil, err
	}

	switch eventType {
	case STREAM_BEGIN_EVENT:
		e := StreamBeginEvent{}
		if err := binary.Read(buf, binary.BigEndian, &e.StreamID); err != nil {
			return nil, err
		}
		return e, nil
	case STREAM_EOF_EVENT:
		e := StreamEOFEvent{}
		if err := binary.Read(buf, binary.BigEndian, &e.StreamID); err != nil {
			return nil, err
		}
		return e, nil
	case STREAM_DRY_EVENT:
		e := StreamDryEvent{}
		if err := binary.Read(buf, binary.BigEndian, &e.StreamID); err != nil {
			return nil, err
		}
		return e, nil
	case SET_BUFFER_LENGTH_EVENT:
		e := SetBufferLengthEvent{}
		if err := binary.Read(buf, binary.BigEndian, &e.StreamID); err != nil {
			return nil, err
		}
		if err := binary.Read(buf, binary.BigEndian, &e.BufferLength); err != nil {
			return nil, err
		}
		return e, nil
	case STREAM_IS_RECORDED_EVENT:
		e := StreamIsRecordedEvent{}
		if err := binary.Read(buf, binary.BigEndian, &e.StreamID); err != nil {
			return nil, err
		}
		return e, nil
	case PING_REQUEST_EVENT:
		e := PingRequestEvent{}
		if err := binary.Read(buf, binary.BigEndian, &e.Timestamp); err != nil {
			return nil, err
		}
		return e, nil
	case PING_RESPONSE_EVENT:
		e := PingResponseEvent{}
		if err := binary.Read(buf, binary.BigEndian, &e.Timestamp); err != nil {
			return nil, err
		}
		return e, nil
	default:
		return nil, fmt.Errorf("Invalid event type: %d", eventType)
	}
}
