// Copyright 2013, zhangpeihao All rights reserved.

package rtmp

import (
	"bytes"
	"encoding/binary"
	"errors"
)

var (
	ErrInvalidControlMessage = errors.New("invalid control message")
)

type Message interface {
	RawMessage() (*RawMessage, error)
}

type Header struct {
	// Basic Header
	Fmt           uint8
	ChunkStreamID uint32

	// Chunk Message Header
	Timestamp       uint32
	MessageLength   uint32
	MessageTypeID   uint8
	MessageStreamID uint32

	// Extended Timestamp
	ExtendedTimestamp uint32

	// Calculated Absolute Timestamp
	AbsoluteTimestamp uint32
}

type RawMessage struct {
	ChunkStreamID uint32
	Timestamp     uint32
	Type          uint8
	StreamID      uint32
	Data          *bytes.Buffer
}

type SetChunkSizeMessage struct {
	Size uint32
}

func ParseControlMessage(raw *RawMessage) (Message, error) {
	if raw.ChunkStreamID != CS_ID_PROTOCOL_CONTROL {
		return nil, ErrInvalidControlMessage
	}

	switch raw.Type {
	case SET_CHUNK_SIZE:
		return ParseSetChunkSizeMessage(raw)
	case ABORT_MESSAGE:
		return ParseAbortMessage(raw)
	case ACKNOWLEDGEMENT:
		return ParseAcknowledgementMessage(raw)
	case WINDOW_ACKNOWLEDGEMENT_SIZE:
		return ParseWindowAcknowledgementSizeMessage(raw)
	case SET_PEER_BANDWIDTH:
		return ParseSetPeerBandwidthMessage(raw)
	default:
		return nil, ErrInvalidControlMessage
	}
}

func ParseSetChunkSizeMessage(raw *RawMessage) (SetChunkSizeMessage, error) {
	var msg SetChunkSizeMessage
	if raw.ChunkStreamID != CS_ID_PROTOCOL_CONTROL {
		return msg, ErrInvalidControlMessage
	}

	if raw.Type != SET_CHUNK_SIZE {
		return msg, ErrInvalidControlMessage
	}

	err := binary.Read(raw.Data, binary.BigEndian, &msg.Size)
	return msg, err
}

func (msg SetChunkSizeMessage) RawMessage() (*RawMessage, error) {
	data := make([]byte, 4)
	binary.BigEndian.PutUint32(data, msg.Size)
	return &RawMessage{
		ChunkStreamID: CS_ID_PROTOCOL_CONTROL,
		Type:          SET_CHUNK_SIZE,
		Data:          bytes.NewBuffer(data),
	}, nil
}

type AbortMessage struct {
	ChunkStreamID uint32
}

func ParseAbortMessage(raw *RawMessage) (AbortMessage, error) {
	var msg AbortMessage
	if raw.ChunkStreamID != CS_ID_PROTOCOL_CONTROL {
		return msg, ErrInvalidControlMessage
	}

	if raw.Type != ABORT_MESSAGE {
		return msg, ErrInvalidControlMessage
	}

	err := binary.Read(raw.Data, binary.BigEndian, &msg.ChunkStreamID)
	return msg, err
}

func (msg AbortMessage) RawMessage() (*RawMessage, error) {
	data := make([]byte, 4)
	binary.BigEndian.PutUint32(data, msg.ChunkStreamID)
	return &RawMessage{
		ChunkStreamID: CS_ID_PROTOCOL_CONTROL,
		Type:          ABORT_MESSAGE,
		Data:          bytes.NewBuffer(data),
	}, nil
}

type AcknowledgementMessage struct {
	Sequence uint32
}

func ParseAcknowledgementMessage(raw *RawMessage) (AcknowledgementMessage, error) {
	var msg AcknowledgementMessage
	if raw.ChunkStreamID != CS_ID_PROTOCOL_CONTROL {
		return msg, ErrInvalidControlMessage
	}

	if raw.Type != ACKNOWLEDGEMENT {
		return msg, ErrInvalidControlMessage
	}

	err := binary.Read(raw.Data, binary.BigEndian, &msg.Sequence)
	return msg, err
}

func (msg AcknowledgementMessage) RawMessage() (*RawMessage, error) {
	data := make([]byte, 4)
	binary.BigEndian.PutUint32(data, msg.Sequence)
	return &RawMessage{
		ChunkStreamID: CS_ID_PROTOCOL_CONTROL,
		Type:          ACKNOWLEDGEMENT,
		Data:          bytes.NewBuffer(data),
	}, nil
}

type WindowAcknowledgementSizeMessage struct {
	WindowSize uint32
}

func ParseWindowAcknowledgementSizeMessage(raw *RawMessage) (WindowAcknowledgementSizeMessage, error) {
	var msg WindowAcknowledgementSizeMessage
	if raw.ChunkStreamID != CS_ID_PROTOCOL_CONTROL {
		return msg, ErrInvalidControlMessage
	}

	if raw.Type != WINDOW_ACKNOWLEDGEMENT_SIZE {
		return msg, ErrInvalidControlMessage
	}

	err := binary.Read(raw.Data, binary.BigEndian, &msg.WindowSize)
	return msg, err
}

func (msg WindowAcknowledgementSizeMessage) RawMessage() (*RawMessage, error) {
	data := make([]byte, 4)
	binary.BigEndian.PutUint32(data, msg.WindowSize)
	return &RawMessage{
		ChunkStreamID: CS_ID_PROTOCOL_CONTROL,
		Type:          WINDOW_ACKNOWLEDGEMENT_SIZE,
		Data:          bytes.NewBuffer(data),
	}, nil
}

type SetPeerBandwidthMessage struct {
	WindowSize uint32
	Type       uint8
}

func ParseSetPeerBandwidthMessage(raw *RawMessage) (SetPeerBandwidthMessage, error) {
	var msg SetPeerBandwidthMessage
	if raw.ChunkStreamID != CS_ID_PROTOCOL_CONTROL {
		return msg, ErrInvalidControlMessage
	}

	if raw.Type != SET_PEER_BANDWIDTH {
		return msg, ErrInvalidControlMessage
	}

	if err := binary.Read(raw.Data, binary.BigEndian, &msg.WindowSize); err != nil {
		return msg, err
	}

	if err := binary.Read(raw.Data, binary.BigEndian, &msg.Type); err != nil {
		return msg, err
	}

	return msg, nil
}

func (msg SetPeerBandwidthMessage) RawMessage() (*RawMessage, error) {
	data := make([]byte, 5)
	binary.BigEndian.PutUint32(data, msg.WindowSize)
	data[4] = uint8(msg.Type)
	return &RawMessage{
		ChunkStreamID: CS_ID_PROTOCOL_CONTROL,
		Type:          SET_PEER_BANDWIDTH,
		Data:          bytes.NewBuffer(data),
	}, nil
}

func (m *RawMessage) RawMessage() (*RawMessage, error) {
	return m, nil
}
