package rtmp

import (
	"code.justin.tv/event-engineering/rtmp/pkg/conn" // Wraps net.Conn setting idleTimout on Write
	amf "code.justin.tv/event-engineering/gortmp/pkg/amf"
	gortmp "code.justin.tv/event-engineering/gortmp/pkg/rtmp"

	"context"
	"fmt"
	"net"
	"net/url"
	"strings"
	"sync"
	"time"
)

// TransactionHandler defines a function to be used as a handler for a given transaction
type TransactionHandler func(gortmp.Message) error

// TransactionMap manages transactions and handlers for those transactions
type TransactionMap struct {
	mu    sync.Mutex
	m     map[uint32]TransactionHandler
	txnid uint32
}

// NewTransactionMap creates a new TransactionMap
func NewTransactionMap() *TransactionMap {
	return &TransactionMap{
		m: make(map[uint32]TransactionHandler),
	}
}

// New registers a handler with a new transaction, returning the transaction ID
func (tm *TransactionMap) New(h TransactionHandler) uint32 {
	tm.mu.Lock()
	defer tm.mu.Unlock()

	id := tm.txnid
	tm.txnid++
	tm.m[id] = h

	return id
}

// Handle uses the supplied transaction ID to call the handler supplied when the transaction was created
func (tm *TransactionMap) Handle(id uint32, msg gortmp.Message) error {
	if h, ok := tm.m[id]; ok {
		delete(tm.m, id)
		return h(msg)
	}
	return nil
}

func ParseMessage(raw *gortmp.RawMessage) (gortmp.Message, error) {
	if raw.ChunkStreamID == gortmp.CS_ID_PROTOCOL_CONTROL {
		var msg gortmp.Message
		var err error

		if raw.Type == gortmp.USER_CONTROL_MESSAGE {
			msg, err = gortmp.ParseEvent(raw)
		} else {
			msg, err = gortmp.ParseControlMessage(raw)
		}

		return msg, err
	}

	switch raw.Type {
	case gortmp.COMMAND_AMF0:
		fallthrough
	case gortmp.COMMAND_AMF3:
		return gortmp.ParseCommand(raw)
	case gortmp.USER_CONTROL_MESSAGE:
		return gortmp.ParseEvent(raw)
	default:
		return raw, nil
	}
}

func Send(conn gortmp.BasicConn, msg gortmp.Message) error {
	raw, err := msg.RawMessage()
	if err != nil {
		return err
	}

	if err := conn.Write(raw); err != nil {
		return err
	}

	return conn.Flush()
}

func Connect(rtmpURL string, readTimeout, writeTimeout time.Duration) (*url.URL, gortmp.BasicConn, string, error) {
	parsed, err := url.Parse(rtmpURL)
	if err != nil {
		return nil, nil, "", fmt.Errorf("Invalid rtmp url: %s", err)
	}

	conn, err := dial(parsed.Host, 0, 0)
	if err != nil {
		return nil, nil, "", err
	}

	path := strings.Split(parsed.Path, "/")
	var appName string

	if len(path) < 2 {
		//return nil, nil, "", fmt.Errorf(`Could not parse app from URL defaulting to "app" %v`, parsed.Path)
		appName = "app"
	} else {
		appName = path[1]
	}

	return parsed, conn, appName, nil
}

func dial(host string, readTimeout, writeTimeout time.Duration) (gortmp.BasicConn, error) {
	if _, _, err := net.SplitHostPort(host); err != nil {
		host = net.JoinHostPort(host, "1935")
	}

	netConn, err := net.Dial("tcp", host)

	if err != nil {
		return nil, err
	}

	if _, err := gortmp.Handshake(context.Background(), netConn); err != nil {
		return nil, fmt.Errorf("Handshake failed: %s", err)
	}

	outputConn := conn.New(netConn, time.Second*readTimeout, time.Second*writeTimeout)

	return gortmp.NewBasicConn(outputConn), nil
}

func DefaultConnectCommand(appName, tcURL string, transactionID uint32) gortmp.ConnectCommand {
	return gortmp.ConnectCommand{
		TransactionID: transactionID,
		Properties: amf.Object{
			"app":           appName,
			"flashVer":      "LNX 9,0,124,2",
			"tcUrl":         tcURL,
			"fpad":          false,
			"capabilities":  15,
			"audioCodecs":   4071,
			"videoCodecs":   252,
			"videoFunction": 1,
		},
	}
}
