package main

import (
	"flag"
	"fmt"
	"log"
	"net"
	"sync"
	"time"

	"context"

	"code.justin.tv/event-engineering/gortmp/pkg/flv"
	"code.justin.tv/event-engineering/gortmp/pkg/rtmp"
)

var listenAddr = flag.String("listen", ":1935", "addr to listen on")
var filename = flag.String("filename", "", "filename to serve")

type handler struct {
}

var _ rtmp.MediaServerHandler = (*handler)(nil)

func (h *handler) Handle(ctx context.Context, r rtmp.Receiver, m rtmp.Message) error {

	return r.Handle(m)
}

func (h *handler) OnMediaStreamCreated(ctx context.Context, ms rtmp.MediaStream) {

}

func (h *handler) OnMediaStreamDestroyed(ctx context.Context, ms rtmp.MediaStream) {

}

type fileStream struct {
	name   string
	tags   []*rtmp.FlvTag
	closed bool
	closer sync.Once
}

var _ rtmp.MediaStream = (*fileStream)(nil)

func (fs *fileStream) Name() string                 { return fs.name }
func (fs *fileStream) Stats() rtmp.MediaStreamStats { return rtmp.MediaStreamStats{} }
func (fs *fileStream) Info() rtmp.MediaStreamInfo   { return rtmp.MediaStreamInfo{} }
func (fs *fileStream) Publish(*rtmp.FlvTag) error   { return fmt.Errorf("invalid operation") }

func (fs *fileStream) Subscribe() (chan *rtmp.FlvTag, error) {
	if fs.closed {
		return nil, fmt.Errorf("stream is closed")
	}

	ch := make(chan *rtmp.FlvTag)
	go func() {
		ref := time.Now()
		for _, t := range fs.tags {
			clock := time.Now().Sub(ref)
			target := time.Millisecond * time.Duration(t.Timestamp)
			waitTime := target - clock

			if waitTime > 500*time.Millisecond {
				// clock drift of >500ms means we need to adjust our reference
				log.Printf("adjusting for clock drift of %s", waitTime)
				ref = ref.Add(-waitTime)
			} else if waitTime > 0 {
				time.Sleep(waitTime)
			}
			ch <- t
		}
		close(ch)
	}()

	return ch, nil
}

func (fs *fileStream) Unsubscribe(ch chan *rtmp.FlvTag) {
	go func() {
		for _ = range ch {
			// just read all the tags and let the channel close
		}
	}()
}

func (fs *fileStream) Wait()          {}
func (fs *fileStream) Close()         { fs.closer.Do(func() { fs.closed = true }) }
func (fs *fileStream) IsClosed() bool { return fs.closed }

func NewFileStream(filePath, name string) (rtmp.MediaStream, error) {
	fs := &fileStream{
		name: name,
	}

	f, err := flv.OpenFile(filePath)
	if err != nil {
		return nil, err
	}
	defer f.Close()

	for !f.IsFinished() {
		header, data, err := f.ReadTag()
		if err != nil {
			return nil, err
		}

		fs.tags = append(fs.tags, &rtmp.FlvTag{
			Type:      header.TagType,
			Timestamp: header.Timestamp,
			Size:      header.DataSize,
			Bytes:     data,
		})
	}

	return fs, nil
}

func main() {
	flag.Parse()
	server := rtmp.NewMediaServer(&handler{})

	ln, err := net.Listen("tcp", *listenAddr)
	if err != nil {
		log.Fatal(err)
	}

	fs, err := NewFileStream(*filename, "file")
	if err != nil {
		log.Fatal(err)
	}

	server.AddStream(context.Background(), fs)

	log.Printf("serving %s at rtmp://%s/app/file", *filename, *listenAddr)

	for {
		conn, err := ln.Accept()
		if err != nil {
			log.Fatal(err)
		}

		go func(conn net.Conn) {
			defer conn.Close()

			log.Print("handshake first")
			if _, err := rtmp.SHandshake(context.Background(), conn); err != nil {
				log.Print(err)
				return
			}

			log.Print("completed handshake")
			log.Print(server.ServeRTMP(context.Background(), conn))
		}(conn)
	}
}
