package codec

import (
	"bytes"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestReadGoodHeader(t *testing.T) {
	assert := assert.New(t)

	testSpecs := []struct {
		header []byte
		want   *AudioSpecificConfig
	}{
		{ // AAC LC Stereo 44100 Hz
			[]byte{0x12, 0x10},
			&AudioSpecificConfig{
				AudioObjectType:           2,
				SamplingFrequencyIndex:    4,
				SamplingFrequency:         44100,
				ChannelConfigurationIndex: 2,
			},
		},
		{ // SBR disabled via sync extension
			[]byte{0x12, 0x10, 0x56, 0xe5, 0x00},
			&AudioSpecificConfig{
				AudioObjectType:           2,
				SamplingFrequencyIndex:    4,
				SamplingFrequency:         44100,
				ChannelConfigurationIndex: 2,
			},
		},
		{ // Trailing zeros
			[]byte{0x12, 0x10, 0x00, 0x00},
			&AudioSpecificConfig{
				AudioObjectType:           2,
				SamplingFrequencyIndex:    4,
				SamplingFrequency:         44100,
				ChannelConfigurationIndex: 2,
			},
		},
		{ // Escape sample rate
			[]byte{0x17, 0x80, 0xf1, 0x20, 0x10},
			&AudioSpecificConfig{
				AudioObjectType:           2,
				SamplingFrequencyIndex:    15,
				SamplingFrequency:         123456,
				ChannelConfigurationIndex: 2,
			},
		},
		{ // PCE channel config
			[]byte{0x11, 0x80, 0x04, 0xc4, 0x00, 0x00, 0x00, 0x00},
			&AudioSpecificConfig{
				AudioObjectType:           2,
				SamplingFrequencyIndex:    3,
				SamplingFrequency:         48000,
				ChannelConfigurationIndex: 0,
			},
		},
	}

	for _, spec := range testSpecs {
		have, err := ReadASC(bytes.NewReader(spec.header))
		assert.Nil(err)
		assert.Equal(have, spec.want)
	}
}

func TestReadBadHeader(t *testing.T) {
	assert := assert.New(t)

	testSpecs := []struct {
		header []byte
	}{
		{ // Too short
			[]byte{},
		},
		{ // Still too short
			[]byte{0x12},
		},
		{ // Bogus sample rate index 14
			[]byte{0x17, 0x10},
		},
		{ // Too short because we are expecting an escape sample rate
			[]byte{0x17, 0x10},
		},
	}

	for _, spec := range testSpecs {
		have, err := ReadASC(bytes.NewReader(spec.header))
		assert.NotNil(err)
		assert.Nil(have)
	}
}

func TestWriteGoodHeader(t *testing.T) {
	assert := assert.New(t)

	testSpecs := []struct {
		in   *AudioSpecificConfig
		want []byte
	}{
		{
			in: &AudioSpecificConfig{
				AudioObjectType:           2,
				SamplingFrequencyIndex:    15,
				SamplingFrequency:         123456,
				ChannelConfigurationIndex: 2,
			},
			want: []byte{0x17, 0x80, 0xf1, 0x20, 0x10},
		},
		{
			in: &AudioSpecificConfig{
				AudioObjectType:           2,
				SamplingFrequencyIndex:    4,
				SamplingFrequency:         44100,
				ChannelConfigurationIndex: 2,
			},
			want: []byte{0x12, 0x10},
		},
	}

	for _, spec := range testSpecs {
		buf := new(bytes.Buffer)
		err := WriteASC(buf, spec.in)
		assert.Nil(err)
		assert.Equal(buf.Bytes(), spec.want)
	}
}
