package flv

import (
	"bytes"
	"errors"
	"io"
	"os"
)

type FlvFileReader struct {
	file      readSeekCloser
	name      string
	size      int64
	headerBuf []byte
}

type TagHeader struct {
	TagType   byte
	DataSize  uint32
	Timestamp uint32
}

type readSeekCloser interface {
	io.Reader
	io.Seeker
	io.Closer
}

func OpenFile(name string) (*FlvFileReader, error) {
	file, err := os.Open(name)
	if err != nil {
		return nil, err
	}

	var size int64
	if size, err = file.Seek(0, 2); err != nil {
		file.Close()
		return nil, err
	}
	if _, err = file.Seek(0, 0); err != nil {
		file.Close()
		return nil, err
	}

	return initFile(file, name, size)
}

type nopCloser struct {
	io.ReadSeeker
}

func (nopCloser) Close() error {
	return nil
}

func WrapInMemoryFile(byteReader *bytes.Reader, name string) (*FlvFileReader, error) {
	return initFile(nopCloser{byteReader}, name, int64(byteReader.Len()))
}

func initFile(file readSeekCloser, name string, size int64) (*FlvFileReader, error) {
	var err error
	flvFile := &FlvFileReader{
		file:      file,
		name:      name,
		size:      size,
		headerBuf: make([]byte, 11),
	}

	// Read flv header
	remain := HEADER_LEN
	flvHeader := make([]byte, remain)

	if _, err = io.ReadFull(file, flvHeader); err != nil {
		file.Close()
		return nil, err
	}
	if flvHeader[0] != 'F' ||
		flvHeader[1] != 'L' ||
		flvHeader[2] != 'V' {
		file.Close()
		return nil, errors.New("FlvFileReader format error")
	}
	return flvFile, nil
}

func (flvFile *FlvFileReader) Close() {
	flvFile.file.Close()
}

func (flvFile *FlvFileReader) Size() (size int64) {
	size = flvFile.size
	return
}

func (flvFile *FlvFileReader) ReadTag() (header *TagHeader, data []byte, err error) {
	tmpBuf := make([]byte, 4)
	header = &TagHeader{}
	// Read tag header
	if _, err = io.ReadFull(flvFile.file, tmpBuf[3:]); err != nil {
		return
	}
	header.TagType = tmpBuf[3]

	// Read tag size
	if _, err = io.ReadFull(flvFile.file, tmpBuf[1:]); err != nil {
		return
	}
	header.DataSize = uint32(tmpBuf[1])<<16 | uint32(tmpBuf[2])<<8 | uint32(tmpBuf[3])

	// Read timestamp
	if _, err = io.ReadFull(flvFile.file, tmpBuf); err != nil {
		return
	}
	header.Timestamp = uint32(tmpBuf[3])<<24 + uint32(tmpBuf[0])<<16 + uint32(tmpBuf[1])<<8 + uint32(tmpBuf[2])

	// Read stream ID
	if _, err = io.ReadFull(flvFile.file, tmpBuf[1:]); err != nil {
		return
	}

	// Read data
	data = make([]byte, header.DataSize)
	if _, err = io.ReadFull(flvFile.file, data); err != nil {
		return
	}

	// Read previous tag size
	if _, err = io.ReadFull(flvFile.file, tmpBuf); err != nil {
		return
	}

	return
}

func (flvFile *FlvFileReader) IsFinished() bool {
	pos, err := flvFile.file.Seek(0, 1)
	return (err != nil) || (pos >= flvFile.size)
}

func (flvFile *FlvFileReader) LoopBack() {
	flvFile.file.Seek(HEADER_LEN, 0)
}

func (flvFile *FlvFileReader) FilePath() string {
	return flvFile.name
}
