package rtmp

import (
	"bufio"
	"encoding/binary"
	"fmt"
	"io"
	"math"
	"net"
	"sync"
)

type windowReader struct {
	io.Reader

	windowSize uint32
	lastWindow uint32
	bytesRead  uint32

	onWindowFull func(uint32) error
}

func (wr *windowReader) Read(buf []byte) (int, error) {
	n, err := wr.Reader.Read(buf)
	if err != nil {
		return n, err
	}

	wr.bytesRead += uint32(n)

	var windowFull bool
	var windowBytes uint32

	// if we hit a rollover point, or we hit the window size
	if wr.bytesRead > math.MaxInt32 {
		windowFull = true
		windowBytes = math.MaxInt32

		wr.bytesRead -= math.MaxInt32
		wr.lastWindow = 0
	} else if wr.bytesRead-wr.lastWindow >= wr.windowSize {
		windowFull = true
		windowBytes = uint32(wr.bytesRead)

		wr.lastWindow = wr.bytesRead
	}

	if windowFull {
		// it's safe to do this synchronously because we're setting our
		// window size to 1/4 of what was sent by the client
		err := wr.onWindowFull(windowBytes)
		if err != nil {
			return n, fmt.Errorf("error sending window acknowledgement: %s", err)
		}
	}

	return n, err
}

// BasicConn reads and writes raw messages
type BasicConn interface {
	Read() (*RawMessage, error)
	Write(*RawMessage) error

	Close() error
	Flush() error

	RemoteAddr() net.Addr
	LocalAddr() net.Addr
}

type basicConn struct {
	netConn net.Conn

	r *bufio.Reader
	w *bufio.Writer

	csr ChunkStreamReader
	csw ChunkStreamWriter

	wr *windowReader

	rMu sync.Mutex
	wMu sync.Mutex
}

func NewBasicConn(conn net.Conn) BasicConn {
	c := &basicConn{
		netConn: conn,
		r:       bufio.NewReader(conn),
		w:       bufio.NewWriter(conn),
	}

	wr := &windowReader{
		Reader:     c.r,
		windowSize: DEFAULT_WINDOW_SIZE,
		onWindowFull: func(size uint32) error {
			msg, err := AcknowledgementMessage{size}.RawMessage()
			if err != nil {
				return err
			}

			if err := c.Write(msg); err != nil {
				return err
			}

			return c.Flush()
		},
	}

	c.csr = NewChunkStreamReader(wr)
	c.csw = NewChunkStreamWriter(c.w)
	c.wr = wr

	return c
}

func (c *basicConn) Read() (*RawMessage, error) {
	c.rMu.Lock()
	defer c.rMu.Unlock()

	raw, err := c.csr.Read()
	if err != nil {
		return raw, err
	}

	// read window ack size mssages and do the right thing
	if raw.ChunkStreamID == CS_ID_PROTOCOL_CONTROL && raw.Type == WINDOW_ACKNOWLEDGEMENT_SIZE && raw.Data.Len() == 4 {
		// other implementations appear to use a smaller window size than set by the client, we're doing so
		// because it allows us to synchronously write window messages (thus simplifying our implementation)
		ws := binary.BigEndian.Uint32(raw.Data.Bytes())
		c.wr.windowSize = ws / 4

		if c.wr.windowSize == 0 {
			c.Close()
			return nil, fmt.Errorf("invalid window size %d received from client", ws)
		}
	}

	return raw, err
}

func (c *basicConn) Write(msg *RawMessage) error {
	c.wMu.Lock()
	defer c.wMu.Unlock()

	return c.csw.Write(msg)
}

func (c *basicConn) Close() error {
	return c.netConn.Close()
}

func (c *basicConn) Flush() error {
	c.wMu.Lock()
	defer c.wMu.Unlock()
	return c.w.Flush()
}

func (c *basicConn) RemoteAddr() net.Addr {
	return c.netConn.RemoteAddr()
}

func (c *basicConn) LocalAddr() net.Addr {
	return c.netConn.LocalAddr()
}
