package rtmp

import (
	"crypto/rand"
	"io"
	"io/ioutil"
	"math"
	"testing"
	"time"
)

func TestWindowReaderWindowSize(t *testing.T) {
	pr, pw := io.Pipe()
	defer pw.Close()

	windowSizes := []uint32{}
	onWindowFull := func(ws uint32) error {
		windowSizes = append(windowSizes, ws)
		return nil
	}

	wr := &windowReader{
		Reader:       pr,
		windowSize:   DEFAULT_WINDOW_SIZE,
		onWindowFull: onWindowFull,
	}

	// copy some bytes to trigger window size messages
	go func() {
		// send 3.5 windows of data
		if _, err := io.CopyN(pw, rand.Reader, DEFAULT_WINDOW_SIZE*7/2); err != nil {
			t.Error(err)
		}

		pw.Close()
	}()

	// time things out in case there's a bug somewhere
	go func() {
		<-time.After(5 * time.Second)
		pw.Close()
	}()

	if _, err := io.Copy(ioutil.Discard, wr); err != nil {
		t.Fatal(err)
	}

	if len(windowSizes) != 3 {
		t.Fatalf("incorrect number of window size message: %#v", windowSizes)
	}

	t.Logf("got window sizes: %#v", windowSizes)
}

func TestWindowReaderOverflow(t *testing.T) {
	pr, pw := io.Pipe()
	defer pw.Close()

	windowSizes := []uint32{}
	onWindowFull := func(ws uint32) error {
		windowSizes = append(windowSizes, ws)
		return nil
	}

	wr := &windowReader{
		Reader:       pr,
		windowSize:   DEFAULT_WINDOW_SIZE,
		onWindowFull: onWindowFull,
		bytesRead:    math.MaxInt32 - (DEFAULT_WINDOW_SIZE / 2),
		lastWindow:   math.MaxInt32 - (DEFAULT_WINDOW_SIZE / 2),
	}

	// copy some bytes to trigger window size messages
	go func() {
		// send 3 windows of data
		if _, err := io.CopyN(pw, rand.Reader, DEFAULT_WINDOW_SIZE*3); err != nil {
			t.Error(err)
		}

		pw.Close()
	}()

	// time things out in case there's a bug somewhere
	go func() {
		<-time.After(5 * time.Second)
		pw.Close()
	}()

	if _, err := io.Copy(ioutil.Discard, wr); err != nil {
		t.Fatal(err)
	}

	if len(windowSizes) != 3 {
		t.Fatalf("incorrect number of window size message: %#v", windowSizes)
	}

	for _, ws := range windowSizes {
		if ws > math.MaxInt32 {
			t.Fatalf("window size should never be greater than %d: got %d", math.MaxInt32, ws)
		}
	}

	t.Logf("got window sizes: %#v", windowSizes)
}
