package rtmp

import (
	"bytes"
	"errors"
	"time"
)

type FlvAudioHeader struct {
	SoundFormat   uint8
	SoundRate     uint8
	SoundSize     uint8
	SoundType     uint8
	AACPacketType uint8
	CodecData     []byte
}

type FlvVideoHeader struct {
	FrameType     uint8
	CodecID       uint8
	AVCPacketType uint8
	CodecData     []byte
}

/*
	FlvTag contains Size to be compatible with the binary format the tag is stored as on the disk when read or written.
*/

type FlvTag struct {
	Type      uint8
	Timestamp uint32
	Size      uint32 //Size of the Bytes Slice
	Bytes     []byte
	// Track arrival time for tags coming in over the network
	ArrivalTime time.Time
}

var (
	ErrPacketTypeMismatch = errors.New("Packet type mismatch")
	ErrTruncatedPacket    = errors.New("Truncated packet")
)

const (
	FlvCodecAVC = 7
	FlvCodecAAC = 10
)

func (f *FlvTag) GetAudioHeader() (*FlvAudioHeader, error) {
	if f.Type != AUDIO_TYPE {
		return nil, ErrPacketTypeMismatch
	}

	header := &FlvAudioHeader{}
	buf := bytes.NewBuffer(f.Bytes)

	bits, err := buf.ReadByte()
	if err != nil {
		return nil, ErrTruncatedPacket
	}
	header.SoundFormat = bits >> 4
	header.SoundRate = (bits >> 2) & 3
	header.SoundSize = (bits >> 1) & 1
	header.SoundType = bits & 1

	if header.SoundFormat == FlvCodecAAC {
		// A zero valued AACPacketType is a Sequence Header. Don't treat a truncated
		// audio packet as a sequence header
		header.AACPacketType, err = buf.ReadByte()
		if err != nil {
			return nil, ErrTruncatedPacket
		}
	}

	header.CodecData = buf.Bytes()

	return header, nil
}

func (f *FlvTag) GetVideoHeader() (*FlvVideoHeader, error) {
	if f.Type != VIDEO_TYPE {
		return nil, ErrPacketTypeMismatch
	}

	header := &FlvVideoHeader{}
	buf := bytes.NewBuffer(f.Bytes)

	bits, err := buf.ReadByte()
	if err != nil {
		return nil, ErrTruncatedPacket
	}

	header.FrameType = bits >> 4
	header.CodecID = bits & 0xF

	if header.CodecID == FlvCodecAVC {
		// A zero valued AVCPacketType is a Sequence Header. Don't treat a truncated
		// video packet as a sequence header
		header.AVCPacketType, err = buf.ReadByte()
		if err != nil {
			return nil, ErrTruncatedPacket
		}

		_ = buf.Next(3) // Composition Time
	}

	header.CodecData = buf.Bytes()

	return header, nil
}

// Checks if a packet is a sequence header, but does NOT check the header for well-formedness
func (h *FlvVideoHeader) IsSequenceHeader() bool {
	return h.CodecID == FlvCodecAVC && h.AVCPacketType == 0
}

// Checks if a packet is a sequence header, but does NOT check the header for well-formedness
func (h *FlvAudioHeader) IsSequenceHeader() bool {
	return h.SoundFormat == FlvCodecAAC && h.AACPacketType == 0
}
