package rtmp

import (
	"bytes"
	"fmt"
	"reflect"
	"testing"

	"context"
)

func sliceCompareHelper(actual []int, ref []int) error {
	if !reflect.DeepEqual(actual, ref) {
		return fmt.Errorf("got %#v expected %#v", actual, ref)
	}
	return nil
}

func TestHeaderLatching(t *testing.T) {
	var sequenceHeaders = []*FlvTag{
		newTag(VIDEO_TYPE, 0, []byte{
			0x17,             // Frame Type, Codec
			0x00,             // AVCPacketType
			0x00, 0x00, 0x00, // CTS
			// AVC Decoder Config:  h.264 profile: Baseline @L1.3
			0x01, 0x42, 0xc0, 0x0d, 0xff, 0xe1, 0x00, 0x18, 0x67, 0x42, 0xc0, 0x0d, 0xd9, 0x01, 0x71, 0xfe, 0x5e, 0x10,
			0x00, 0x00, 0x03, 0x00, 0x10, 0x00, 0x00, 0x03, 0x03, 0xc0, 0xf1, 0x42, 0xa4, 0x80, 0x01, 0x00, 0x05, 0x68,
			0xcb, 0x83, 0xcb, 0x20,
		}),
		newTag(AUDIO_TYPE, 0, []byte{
			0xaf,                         // FLV AAC Header
			0x00,                         // AACPacketType
			0x12, 0x10, 0x56, 0xe5, 0x00, // AAC LC Stereo 44100 Hz no-SBR sequence header
		}),
		newTag(VIDEO_TYPE, 100, []byte{
			0x17,             // Frame Type, Codec
			0x00,             // AVCPacketType
			0x00, 0x00, 0x00, // CTS
			// AVC Decoder Config: h.264 profile: Main @L1.3
			0x01, 0x4d, 0x40, 0x0d, 0xff, 0xe1, 0x00, 0x18, 0x67, 0x4d, 0x40, 0x0d, 0xec, 0xa0, 0xb8, 0xff, 0x2f, 0x08,
			0x00, 0x00, 0x03, 0x00, 0x08, 0x00, 0x00, 0x03, 0x01, 0xe0, 0x78, 0xa1, 0x4c, 0xb0, 0x01, 0x00, 0x05, 0x68,
			0xeb, 0xe3, 0xcb, 0x20,
		}),
		newTag(AUDIO_TYPE, 100, []byte{
			0xaf,                         // FLV AAC Header
			0x00,                         // AACPacketType
			0x11, 0x90, 0x56, 0xe5, 0x00, // AAC LC Stereo 48000 Hz no-SBR sequence header
		}),
	}
	ctx := context.Background()
	st := NewMediaStream(ctx, "TestHeaderLatching")

	tagWatcher := func(outCh chan<- []int) {
		subCh, err := st.Subscribe()
		if err != nil {
			t.Fatal(err)
		}
		tagsGotten := make([]int, len(sequenceHeaders))
		go func() {
			for tag := range subCh {
				for i, refHdr := range sequenceHeaders {
					if bytes.Equal(tag.Bytes, refHdr.Bytes) {
						tagsGotten[i]++
					}
				}
			}
			outCh <- tagsGotten
		}()
	}

	outChEarly := make(chan []int)
	tagWatcher(outChEarly)

	st.Publish(sequenceHeaders[0])
	st.Publish(sequenceHeaders[1])
	st.Publish(sequenceHeaders[2])
	st.Publish(sequenceHeaders[3])

	outChLate := make(chan []int)
	tagWatcher(outChLate)

	st.Close()

	early := <-outChEarly
	if err := sliceCompareHelper(early, []int{1, 1, 1, 1}); err != nil {
		t.Fatalf("The early subscriber: %s", err.Error())
	}
	late := <-outChLate
	if err := sliceCompareHelper(late, []int{0, 0, 1, 1}); err != nil {
		t.Fatalf("The late subscriber: %s", err.Error())
	}
}

func TestInfo(t *testing.T) {
	ctx := context.Background()
	st := NewMediaStream(ctx, "TestInfo")
	var tags = []*FlvTag{
		newTag(VIDEO_TYPE, 0, []byte{
			0x17,             // Frame Type, Codec
			0x00,             // AVCPacketType
			0x00, 0x00, 0x00, // CTS
			// AVC Decoder Config:  h.264 profile: Baseline @L1.3
			0x01, 0x42, 0xc0, 0x0d, 0xff, 0xe1, 0x00, 0x18, 0x67, 0x42, 0xc0, 0x0d, 0xd9, 0x01, 0x71, 0xfe, 0x5e, 0x10,
			0x00, 0x00, 0x03, 0x00, 0x10, 0x00, 0x00, 0x03, 0x03, 0xc0, 0xf1, 0x42, 0xa4, 0x80, 0x01, 0x00, 0x05, 0x68,
			0xcb, 0x83, 0xcb, 0x20,
		}),
		newTag(AUDIO_TYPE, 0, []byte{
			0xaf,                         // FLV AAC Header
			0x00,                         // AACPacketType
			0x12, 0x10, 0x56, 0xe5, 0x00, // AAC LC Stereo 44100 Hz no-SBR sequence header
		}),
		newTag(VIDEO_TYPE, 0, []byte{
			0x17,             // Frame Type, Codec
			0x01,             // AVCPacketType
			0x00, 0x00, 0x00, // CTS
			// NALUs go here
		}),
		newTag(AUDIO_TYPE, 0, []byte{
			0xaf, // FLV AAC Header
			0x01, // AACPacketType
			// Payload goes here
		}),
		newTag(VIDEO_TYPE, 10, []byte{
			0x17,             // Frame Type, Codec
			0x01,             // AVCPacketType
			0x00, 0x00, 0x00, // CTS
			// NALUs go here
		}),
		newTag(AUDIO_TYPE, 10, []byte{
			0xaf, // FLV AAC Header
			0x01, // AACPacketType
			// Payload goes here
		}),
		newTag(VIDEO_TYPE, 20, []byte{
			0x17,             // Frame Type, Codec
			0x01,             // AVCPacketType
			0x00, 0x00, 0x00, // CTS
			// NALUs go here
		}),
		newTag(AUDIO_TYPE, 20, []byte{
			0xaf, // FLV AAC Header
			0x01, // AACPacketType
			// Payload goes here
		}),
	}
	for _, tag := range tags {
		st.Publish(tag)
	}
	info := st.Info()
	if expected := (MediaStreamInfo{
		VideoHeader:     tags[0],
		AudioHeader:     tags[1],
		LastVideoPacket: tags[len(tags)-2],
		LastAudioPacket: tags[len(tags)-1],
	}); info != expected {
		t.Fatalf("Info() %#v does not match expected %#v", info, expected)
	}
	st.Close()
	info = st.Info()
	if info != (MediaStreamInfo{}) {
		t.Fatalf("Got non empty Info() on closed stream %#v", info)
	}
}
