package main

import (
	"encoding/json"
	"io/ioutil"
	"log"
	"net"
	"net/http"
	"regexp"
)

const (
	clientID = "1xxoh9qrrzy35swpuq4cuyjcra0cxe"
)

var (
	serverRegex = regexp.MustCompile(`^rtmp://([^\.]+)\.twitch.tv/.*$`)
	offnets     = []string{"live-eze", "live-gig", "live-gru", "live-scl", "live-tpe", "live-mil", "live-lis", "live-qro", "live-mde", "live-lim"}
)

type response struct {
	Ingests []item `json:"ingests"`
}

type item struct {
	URLTemplate string `json:"url_template"`
	Name        string `json:"name"`
}

func main() {

	req, err := http.NewRequest("GET", "https://api.twitch.tv/kraken/ingests", nil)
	if err != nil {
		log.Fatal("Could not retrieve ingest list", err)
	}

	req.Header.Set("Accept", "application/vnd.twitchtv.v5+json")
	req.Header.Set("Client-Id", clientID)

	c := new(http.Client)
	resp, err := c.Do(req)
	defer resp.Body.Close()

	if err != nil {
		log.Fatal("Could not retrieve ingest list", err)
	}

	bodyBytes, err := ioutil.ReadAll(resp.Body)

	if err != nil {
		log.Fatal("Could not retrieve ingest list", err)
	}

	ingestResponse := &response{}
	err = json.Unmarshal(bodyBytes, ingestResponse)

	if err != nil {
		log.Fatal("Could not retrieve ingest list", err)
	}

	for _, ingest := range ingestResponse.Ingests {
		matches := serverRegex.FindStringSubmatch(ingest.URLTemplate)
		if len(matches) < 2 || matches[1] == "" {
			log.Println("Malformed URL Template in response", ingest.Name, ingest.URLTemplate)
			continue
		}

		isOffnet := false
		for _, offnet := range offnets {
			if offnet == matches[1] {
				isOffnet = true
				break
			}
		}

		if isOffnet {
			continue
		}

		nonPrivDNSName := matches[1] + ".twitch.tv"
		privDNSName := matches[1] + "-priv.twitch.tv"

		nonPrivIPs, err := net.LookupIP(nonPrivDNSName)
		if err != nil {
			log.Println("Could not look up DNS name", nonPrivDNSName, err)
		}

		privIPs, err := net.LookupIP(privDNSName)
		if err != nil {
			log.Println("Could not look up DNS name", privDNSName, err)
		}

		for _, privIP := range privIPs {
			conn, err := net.Dial("tcp", privIP.String()+":1935")
			if err != nil {
				log.Println("Could not connect to server", privDNSName, err)
			} else {
				conn.Close()
			}
			for _, nonPrivIP := range nonPrivIPs {
				if privIP.Equal(nonPrivIP) {
					log.Printf("Found duplicate IP %v in %v", privIP, nonPrivDNSName)
				} else {
					conn, err := net.Dial("tcp", nonPrivIP.String()+":1935")
					if err != nil {
						log.Println("Could not connect to server", nonPrivDNSName, err)
					} else {
						conn.Close()
					}
				}
			}
		}
	}
}
