package main

import (
	"net/http"
	"os"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/sirupsen/logrus"

	twLambda "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	"code.justin.tv/event-engineering/acm-ca-go/pkg/certgen"
	moonlightAWS "code.justin.tv/event-engineering/moonlight-api/pkg/aws"
	awsBackend "code.justin.tv/event-engineering/moonlight-api/pkg/aws/backend"
	db "code.justin.tv/event-engineering/moonlight-api/pkg/db"
	"code.justin.tv/event-engineering/moonlight-api/pkg/moonlight"
	adminRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/admin"
	"code.justin.tv/video/brassclient"
)

var (
	moonlightAPI *moonlight.AdminAPI
)

func init() {
	logger := logrus.New()
	logger.SetLevel(logrus.DebugLevel)

	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})

	if err != nil {
		panic(err)
	}

	ab := awsBackend.New(sess)

	instancesTable := os.Getenv("instancesTableName")
	serversTable := os.Getenv("serversTableName")
	usersTable := os.Getenv("usersTableName")
	rtmpSourcesTable := os.Getenv("rtmpSourcesTableName")
	jobRunnerQueueURL := os.Getenv("jobRunnerQueueURL")
	rtmpSourceURL := os.Getenv("rtmpSourceURL")
	ecrRepositoryURL := os.Getenv("ecrRepositoryURL")
	publicMediaURL := os.Getenv("publicMediaURL")
	moonlightRootCAARN := os.Getenv("moonlightRootCAARN")

	logger.Info("Creating moonlight db")

	mdb := db.NewDynamoMoonlightDB(ab, instancesTable, serversTable, usersTable, rtmpSourcesTable, logger)
	awsClient := moonlightAWS.New(ab, logger)

	certGen := certgen.New(sess, moonlightRootCAARN)

	brassClient := brassclient.New(sess)
	bindleLockConfig := moonlight.BindleLockConfig{
		CanAccessSystemBindleLockID: os.Getenv("canAccessSystemBindleLockID"),
		AdminBindleLockID:           os.Getenv("adminBindleLockID"),
		OpsBindleLockID:             os.Getenv("opsBindleLockID"),
	}

	moonlightAPI, err = moonlight.NewAdminServer(mdb, awsClient, certGen, jobRunnerQueueURL, rtmpSourceURL, ecrRepositoryURL, publicMediaURL, brassClient, bindleLockConfig, logger)

	if err != nil {
		panic(err)
	}
}

func main() {
	// Create a Twirp service based off the internal service
	err := twLambda.ListenAndServe(":80", &headerMapper{
		baseHandler: adminRPC.NewAdminControllerServer(moonlightAPI, nil),
	})

	if err != nil {
		panic(err)
	}
}

// This will map certain http headers to context values so we can get to them inside our service
type headerMapper struct {
	baseHandler http.Handler
}

func (hm *headerMapper) ServeHTTP(w http.ResponseWriter, req *http.Request) {
	ctx := req.Context()

	userIdentifier := req.Header.Get(moonlight.UserIDHeader)
	ctx = moonlight.WithUserID(ctx, userIdentifier)

	req = req.WithContext(ctx)

	hm.baseHandler.ServeHTTP(w, req)
}
