package main

import (
	awsBackend "code.justin.tv/event-engineering/moonlight-api/pkg/aws/backend"
	"code.justin.tv/event-engineering/moonlight-api/pkg/twitch"
	twitchBackend "code.justin.tv/event-engineering/moonlight-api/pkg/twitch/backend"
	"context"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/sirupsen/logrus"
	"net/http"
	"os"

	apigw "code.justin.tv/coleiain/api-gateway-handler"
	ext "code.justin.tv/event-engineering/moonlight-api/pkg/external"
	"encoding/json"
)

var binaryContentTypes = []string{}

var (
	extAPI http.Handler
	logger *logrus.Logger
)

func init() {
	logger = logrus.New()
	logger.SetLevel(logrus.InfoLevel)
	logger.SetNoLock()

	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})

	if err != nil {
		panic(err)
	}

	ab := awsBackend.New(sess)

	clientID := os.Getenv("clientId")

	tb := twitchBackend.New(clientID, logger)
	twitchClient := twitch.New(tb, logger)

	internalLambdaARN := os.Getenv("internalLambdaARN")

	extAPI = ext.New(internalLambdaARN, ab, twitchClient, logger)
}

func main() {
	lambda.Start(Handle)
}

func Handle(ctx context.Context, request events.APIGatewayProxyRequest) (events.APIGatewayProxyResponse, error) {
	// The underlying code for the API is wired up as an HTTP handler so we need to convert from the API Gateway Proxy Request to a standard http request
	// Since we don't have a way of accessing the RequestContext we're going to grab what we need from it and add it as a custom header making sure to always set it
	// so that it can't be passed through by the client

	authoriser, err := json.Marshal(request.RequestContext.Authorizer)
	if err != nil {
		logger.Warnf("Failed to marshall authoriser %v | %v", err, request.RequestContext.Authorizer)
		return events.APIGatewayProxyResponse{}, err
	}

	// This is always set here so that it can't be passed in by the client
	request.Headers[ext.AuthorizerHeaderName] = string(authoriser)

	return apigw.ProcessRequest(ctx, request, binaryContentTypes, extAPI)
}
