package main

import (
	authoriser "code.justin.tv/event-engineering/moonlight-api/pkg/authoriser"
	twitchBackend "code.justin.tv/event-engineering/moonlight-api/pkg/twitch/backend"
	"context"
	"errors"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/sirupsen/logrus"
	"os"
	"regexp"
)

var (
	twitch      authoriser.Client
	twitchDev   authoriser.Client
	logger      *logrus.Logger
	bearerRegex = regexp.MustCompile("^Bearer\\s([^\\s]+).*$")
)

func init() {
	logger = logrus.New()
	logger.SetLevel(logrus.InfoLevel)
	logger.SetNoLock()

	clientID := os.Getenv("clientId")
	devClientID := os.Getenv("devClientId") // This will be empty on prod

	tb := twitchBackend.New(clientID, logger)
	twitch = authoriser.New(tb, logger)

	// On our staging environment we're also exposing an api stage called /dev that can be used when working locally
	// We need to use a different oauth client for it, so this is how we're handling it
	if devClientID != "" {
		tdb := twitchBackend.New(devClientID, logger)
		twitchDev = authoriser.New(tdb, logger)
	}
}

func main() {
	lambda.Start(Handle)
}

func Handle(ctx context.Context, request events.APIGatewayCustomAuthorizerRequestTypeRequest) (events.APIGatewayCustomAuthorizerResponse, error) {
	// Extract token from header
	token := bearerRegex.ReplaceAllString(request.Headers["Authorization"], "$1")

	var twitchClient = twitch

	logger.Warnf("Checking stage from request context [%v]", request.RequestContext.Stage)

	if request.RequestContext.Stage == "dev" {
		if twitchDev == nil {
			err := errors.New("Stage was dev, but devClient was nil")
			logger.Error(err)
			return events.APIGatewayCustomAuthorizerResponse{}, err
		}
		twitchClient = twitchDev
	}

	// TODO work out the right thing here depending on how responses from the oauth validate call work
	uid, err := twitchClient.GetUserIdentity(token)
	if err != nil {
		logger.Warnf("Could not get user identity %v", err)
		return events.APIGatewayCustomAuthorizerResponse{}, errors.New("Unauthorized")
	}

	return events.APIGatewayCustomAuthorizerResponse{
		PrincipalID: uid.UserId,
		PolicyDocument: events.APIGatewayCustomAuthorizerPolicy{
			Version: "2012-10-17",
			Statement: []events.IAMPolicyStatement{
				{
					Action:   []string{"execute-api:Invoke"},
					Effect:   "Allow",
					Resource: []string{request.MethodArn},
				},
			},
		},
		// this is used by the api to identify the user
		Context: map[string]interface{}{
			"user_id":    uid.UserId,
			"user_name":  uid.UserName,
			"token_used": token,
		},
	}, nil
}
