package aws

import (
	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	"code.justin.tv/event-engineering/moonlight-api/pkg/aws/backend"
	"encoding/json"
	"github.com/aws/aws-sdk-go/service/autoscaling"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/ec2"
	"github.com/aws/aws-sdk-go/service/lambda/lambdaiface"
	"github.com/aws/aws-sdk-go/service/sqs"
)

var dataTypeString = "String"

// New generates a new AWS Client with the supplied backend client
func New(awsClient backend.Client, logger logging.Logger) *Client {
	return &Client{
		aws:    awsClient,
		logger: logger,
	}
}

// Client is an AWS client containing methods for use in the moonlight system
type Client struct {
	aws    backend.Client
	logger logging.Logger
}

// GetLambdaClient returns the lambda client for use in twirp lambda transport
func (c *Client) GetLambdaClient() lambdaiface.LambdaAPI {
	return c.aws.GetLambdaClient()
}

func (c *Client) SendSQSJob(queueURL, jobType string, delay int64, payload interface{}) error {

	jobTypeAttr := sqs.MessageAttributeValue{
		DataType:    &dataTypeString,
		StringValue: &jobType,
	}

	data, err := json.Marshal(payload)

	if err != nil {
		return err
	}

	stringData := string(data)

	input := sqs.SendMessageInput{
		QueueUrl:     &queueURL,
		DelaySeconds: &delay,
		MessageAttributes: map[string]*sqs.MessageAttributeValue{
			"job_type": &jobTypeAttr,
		},
		MessageBody: &stringData,
	}

	output, err := c.aws.SQSSendMessage(&input)

	if err != nil {
		return err
	}

	c.logger.Debugf("Created message with ID %v", output.MessageId)

	return nil
}

func (c *Client) SQSSendMessage(input *sqs.SendMessageInput) (*sqs.SendMessageOutput, error) {
	return c.aws.SQSSendMessage(input)
}

// DDBGetItem calls the underlying DDBGetItem from the aws backend
func (c *Client) DDBGetItem(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
	return c.aws.DDBGetItem(input)
}

// DDBPutItem calls the underlying DDBPutItem from the aws backend
func (c *Client) DDBPutItem(input *dynamodb.PutItemInput) (*dynamodb.PutItemOutput, error) {
	return c.aws.DDBPutItem(input)
}

// DDBUpdateItem calls the underlying DDBUpdateItem from the aws backend
func (c *Client) DDBUpdateItem(input *dynamodb.UpdateItemInput) (*dynamodb.UpdateItemOutput, error) {
	return c.aws.DDBUpdateItem(input)
}

// DDBDeleteItem calls the underlying DDBDeleteItem from the aws backend
func (c *Client) DDBDeleteItem(input *dynamodb.DeleteItemInput) (*dynamodb.DeleteItemOutput, error) {
	return c.aws.DDBDeleteItem(input)
}

// DDBQuery calls the underlying DDBQuery from the aws backend
func (c *Client) DDBQuery(input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error) {
	return c.aws.DDBQuery(input)
}

// DDBScan calls the underlying DDBScan from the aws backend
func (c *Client) DDBScan(input *dynamodb.ScanInput) (*dynamodb.ScanOutput, error) {
	return c.aws.DDBScan(input)
}

// EC2DescribeInstances calls the underlying EC2DescribeInstances from the aws backend
func (c *Client) EC2DescribeInstances(input *ec2.DescribeInstancesInput) (*ec2.DescribeInstancesOutput, error) {
	return c.aws.EC2DescribeInstances(input)
}

// AutoScalingTerminateInstance calls the underlying AutoScalingTerminateInstance from the aws backend
func (c *Client) AutoScalingTerminateInstance(input *autoscaling.TerminateInstanceInAutoScalingGroupInput) (*autoscaling.TerminateInstanceInAutoScalingGroupOutput, error) {
	return c.aws.AutoScalingTerminateInstance(input)
}

// AutoScalingSetDesiredCapacity calls the underlying AutoScalingSetDesiredCapacity from the aws backend
func (c *Client) AutoScalingSetDesiredCapacity(input *autoscaling.SetDesiredCapacityInput) (*autoscaling.SetDesiredCapacityOutput, error) {
	return c.aws.AutoScalingSetDesiredCapacity(input)
}
