package backend

import (
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/autoscaling"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/ec2"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/aws/aws-sdk-go/service/lambda/lambdaiface"
	"github.com/aws/aws-sdk-go/service/sqs"
)

// Client contains methods for AWS interactions to allow easier data mocking for tests.
// If this interface changes, counterfeiter must be re-run using `make mocks`
type Client interface {
	DDBGetItem(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error)
	DDBPutItem(input *dynamodb.PutItemInput) (*dynamodb.PutItemOutput, error)
	DDBUpdateItem(input *dynamodb.UpdateItemInput) (*dynamodb.UpdateItemOutput, error)
	DDBDeleteItem(input *dynamodb.DeleteItemInput) (*dynamodb.DeleteItemOutput, error)
	DDBQuery(input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error)
	DDBScan(input *dynamodb.ScanInput) (*dynamodb.ScanOutput, error)
	SQSSendMessage(input *sqs.SendMessageInput) (*sqs.SendMessageOutput, error)
	EC2DescribeInstances(input *ec2.DescribeInstancesInput) (*ec2.DescribeInstancesOutput, error)
	AutoScalingTerminateInstance(input *autoscaling.TerminateInstanceInAutoScalingGroupInput) (*autoscaling.TerminateInstanceInAutoScalingGroupOutput, error)
	AutoScalingSetDesiredCapacity(input *autoscaling.SetDesiredCapacityInput) (*autoscaling.SetDesiredCapacityOutput, error)
	GetLambdaClient() lambdaiface.LambdaAPI
}

type client struct {
	ddb    *dynamodb.DynamoDB
	sqs    *sqs.SQS
	ec2    *ec2.EC2
	asg    *autoscaling.AutoScaling
	lambda *lambda.Lambda
}

// New generates all the AWS clients required for this service using chain credentials from environment then IAM role
func New(sess *session.Session) Client {
	return &client{
		ddb:    dynamodb.New(sess),
		sqs:    sqs.New(sess),
		ec2:    ec2.New(sess),
		asg:    autoscaling.New(sess),
		lambda: lambda.New(sess),
	}
}

// GetLambdaClient returns the lambda client for use in twirp lambda transport
func (c *client) GetLambdaClient() lambdaiface.LambdaAPI {
	return c.lambda
}

// DDBGetItem calls the underlying GetItem from the dynamodb backend
func (c *client) DDBGetItem(input *dynamodb.GetItemInput) (*dynamodb.GetItemOutput, error) {
	return c.ddb.GetItem(input)
}

// DDBPutItem calls the underlying PutItem from the dynamodb backend
func (c *client) DDBPutItem(input *dynamodb.PutItemInput) (*dynamodb.PutItemOutput, error) {
	return c.ddb.PutItem(input)
}

// DDBUpdateItem calls the underlying UpdateItem from the dynamodb backend
func (c *client) DDBUpdateItem(input *dynamodb.UpdateItemInput) (*dynamodb.UpdateItemOutput, error) {
	return c.ddb.UpdateItem(input)
}

// DDBDeleteItem calls the underlying DeleteItem from the dynamodb backend
func (c *client) DDBDeleteItem(input *dynamodb.DeleteItemInput) (*dynamodb.DeleteItemOutput, error) {
	return c.ddb.DeleteItem(input)
}

// DDBQuery calls the underlying Query from the dynamodb backend
func (c *client) DDBQuery(input *dynamodb.QueryInput) (*dynamodb.QueryOutput, error) {
	return c.ddb.Query(input)
}

// DDBScan calls the underlying Scan from the dynamodb backend
func (c *client) DDBScan(input *dynamodb.ScanInput) (*dynamodb.ScanOutput, error) {
	return c.ddb.Scan(input)
}

// SQSSendMessage calls the underlying SendMessage from the sqs backend
func (c *client) SQSSendMessage(input *sqs.SendMessageInput) (*sqs.SendMessageOutput, error) {
	return c.sqs.SendMessage(input)
}

// EC2DescribeInstances calls the underlying DescribeInstances from the ec2 backend
func (c *client) EC2DescribeInstances(input *ec2.DescribeInstancesInput) (*ec2.DescribeInstancesOutput, error) {
	return c.ec2.DescribeInstances(input)
}

// AutoScalingTerminateInstance calls the underlying TerminateInstanceInAutoScalingGroup from the autoscaling backend
func (c *client) AutoScalingTerminateInstance(input *autoscaling.TerminateInstanceInAutoScalingGroupInput) (*autoscaling.TerminateInstanceInAutoScalingGroupOutput, error) {
	return c.asg.TerminateInstanceInAutoScalingGroup(input)
}

// AutoScalingSetDesiredCapacity calls the underlying SetDesiredCapacity from the autoscaling backend
func (c *client) AutoScalingSetDesiredCapacity(input *autoscaling.SetDesiredCapacityInput) (*autoscaling.SetDesiredCapacityOutput, error) {
	return c.asg.SetDesiredCapacity(input)
}
