package db

import (
	"code.justin.tv/event-engineering/golibs/pkg/logging"
	awsBackend "code.justin.tv/event-engineering/moonlight-api/pkg/aws/backend"
	"errors"
	"fmt"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

type ddbStateStore struct {
	aws              awsBackend.Client
	instancesTable   string
	serversTable     string
	usersTable       string
	rtmpSourcesTable string
	logger           logging.Logger
}

// NewDynamoMoonlightDB generates a new moonlight DB using the dynamodb backend
func NewDynamoMoonlightDB(aws awsBackend.Client, instancesTable, serversTable, usersTable, rtmpSourcesTable string, logger logging.Logger) MoonlightDB {
	return &ddbStateStore{
		aws:              aws,
		instancesTable:   instancesTable,
		serversTable:     serversTable,
		usersTable:       usersTable,
		rtmpSourcesTable: rtmpSourcesTable,
		logger:           logger,
	}
}

func (d *ddbStateStore) AddInstance(instance *Instance) error {
	item, err := dynamodbattribute.MarshalMap(instance)
	if err != nil {
		return err
	}

	input := &dynamodb.PutItemInput{
		TableName: &d.instancesTable,
		Item:      item,
	}

	_, err = d.aws.DDBPutItem(input)

	return err
}

func (d *ddbStateStore) GetInstance(ID string) (*Instance, error) {
	input := &dynamodb.GetItemInput{
		TableName: &d.instancesTable,
		Key: map[string]*dynamodb.AttributeValue{
			"instance_id": &dynamodb.AttributeValue{
				S: &ID,
			},
		},
	}

	output, err := d.aws.DDBGetItem(input)
	if err != nil {
		return nil, err
	}

	if len(output.Item) == 0 {
		return nil, errors.New("Instance Not Found")
	}

	instance := &Instance{}

	err = dynamodbattribute.UnmarshalMap(output.Item, instance)
	if err != nil {
		return nil, err
	}

	return instance, nil
}

func (d *ddbStateStore) ListInstances() ([]*Instance, error) {
	d.logger.Debug("ListInstances: start")
	input := &dynamodb.ScanInput{
		TableName: &d.instancesTable,
	}

	d.logger.Debugf("ListInstances: scanning %v", d.instancesTable)

	output, err := d.aws.DDBScan(input)
	if err != nil {
		d.logger.Warnf("Error scanning dynamo db: %v", err)
		return nil, err
	}

	ret := make([]*Instance, 0)

	for _, item := range output.Items {
		instance := &Instance{}
		err := dynamodbattribute.UnmarshalMap(item, instance)
		if err != nil {
			return nil, err
		}
		ret = append(ret, instance)
	}

	d.logger.Debug("ListInstances: end")

	return ret, nil
}

func (d *ddbStateStore) ListInstancesByServer(serverID string) ([]*Instance, error) {
	d.logger.Debug("ListInstancesByServer: start")

	filter := "server_id = :server_id"
	index := "ServerIdIndex"

	input := &dynamodb.QueryInput{
		TableName:              &d.instancesTable,
		IndexName:              &index,
		KeyConditionExpression: &filter,
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":server_id": &dynamodb.AttributeValue{S: &serverID},
		},
	}

	output, err := d.aws.DDBQuery(input)
	if err != nil {
		d.logger.Warnf("Error querying dynamo db: %v", err)
		return nil, err
	}

	d.logger.Debugf("ListInstancesByServer: got %v items", len(output.Items))

	ret := make([]*Instance, 0)

	for _, item := range output.Items {
		instance := &Instance{}
		err := dynamodbattribute.UnmarshalMap(item, instance)
		if err != nil {
			return nil, err
		}
		ret = append(ret, instance)
	}

	d.logger.Debug("ListInstancesByServer: end")

	return ret, nil
}

func (d *ddbStateStore) ListInstancesByRtmpKey(rtmpKey string) ([]*Instance, error) {
	d.logger.Debug("ListInstancesByRtmpKey: start")

	filter := "rtmp_key = :rtmp_key"
	index := "RtmpKeyIndex"

	input := &dynamodb.QueryInput{
		TableName:              &d.instancesTable,
		IndexName:              &index,
		KeyConditionExpression: &filter,
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":rtmp_key": &dynamodb.AttributeValue{S: &rtmpKey},
		},
	}

	output, err := d.aws.DDBQuery(input)
	if err != nil {
		d.logger.Warnf("Error querying dynamo db: %v", err)
		return nil, err
	}

	d.logger.Debugf("ListInstancesByRtmpKey: got %v items", len(output.Items))

	ret := make([]*Instance, 0)

	for _, item := range output.Items {
		instance := &Instance{}
		err := dynamodbattribute.UnmarshalMap(item, instance)
		if err != nil {
			return nil, err
		}
		ret = append(ret, instance)
	}

	d.logger.Debug("ListInstancesByRtmpKey: end")

	return ret, nil
}

func (d *ddbStateStore) ListInstancesByTwitchUserID(twitchUserID string) ([]*Instance, error) {
	d.logger.Debug("ListInstancesByTwitchUserID: start")

	filter := "twitch_user_id = :twitch_user_id"
	index := "TwitchUserIDIndex"

	input := &dynamodb.QueryInput{
		TableName:              &d.instancesTable,
		IndexName:              &index,
		KeyConditionExpression: &filter,
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":twitch_user_id": &dynamodb.AttributeValue{S: &twitchUserID},
		},
	}

	output, err := d.aws.DDBQuery(input)
	if err != nil {
		d.logger.Warnf("Error querying dynamo db: %v", err)
		return nil, err
	}

	d.logger.Debugf("ListInstancesByTwitchUserID: got %v items", len(output.Items))

	ret := make([]*Instance, 0)

	for _, item := range output.Items {
		instance := &Instance{}
		err := dynamodbattribute.UnmarshalMap(item, instance)
		if err != nil {
			return nil, err
		}
		ret = append(ret, instance)
	}

	d.logger.Debug("ListInstancesByTwitchUserID: end")

	return ret, nil
}

func (d *ddbStateStore) RemoveInstance(ID string) error {
	input := &dynamodb.DeleteItemInput{
		TableName: &d.instancesTable,
		Key: map[string]*dynamodb.AttributeValue{
			"instance_id": &dynamodb.AttributeValue{S: &ID},
		},
	}

	_, err := d.aws.DDBDeleteItem(input)

	return err
}

func (d *ddbStateStore) UpdateInstance(instance *Instance) error {
	item, err := dynamodbattribute.MarshalMap(instance)
	if err != nil {
		return err
	}

	input := &dynamodb.PutItemInput{
		TableName: &d.instancesTable,
		Item:      item,
	}

	// For DDB put is update for our purposes
	_, err = d.aws.DDBPutItem(input)

	return err
}

func (d *ddbStateStore) AddDaemon(daemon *Daemon) error {
	item, err := dynamodbattribute.MarshalMap(daemon)
	if err != nil {
		return err
	}

	condition := "server_id <> :server_id"

	input := &dynamodb.PutItemInput{
		TableName:           &d.serversTable,
		Item:                item,
		ConditionExpression: &condition,
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":server_id": &dynamodb.AttributeValue{S: &daemon.ID},
		},
	}

	_, err = d.aws.DDBPutItem(input)

	return err
}

func (d *ddbStateStore) GetDaemon(ID string) (*Daemon, error) {
	input := &dynamodb.GetItemInput{
		TableName: &d.serversTable,
		Key: map[string]*dynamodb.AttributeValue{
			"server_id": &dynamodb.AttributeValue{S: &ID},
		},
	}

	item, err := d.aws.DDBGetItem(input)

	if err != nil {
		return nil, err
	}

	if len(item.Item) == 0 {
		return nil, errors.New("Daemon Not Found")
	}

	ret := &Daemon{}

	err = dynamodbattribute.UnmarshalMap(item.Item, ret)

	if err != nil {
		return nil, err
	}

	return ret, nil
}

func (d *ddbStateStore) RemoveDaemon(ID string) error {
	input := &dynamodb.DeleteItemInput{
		TableName: &d.serversTable,
		Key: map[string]*dynamodb.AttributeValue{
			"server_id": &dynamodb.AttributeValue{S: &ID},
		},
	}

	_, err := d.aws.DDBDeleteItem(input)

	return err
}

func (d *ddbStateStore) UpdateDaemon(daemon *Daemon) error {
	item, err := dynamodbattribute.MarshalMap(daemon)
	if err != nil {
		return err
	}

	input := &dynamodb.PutItemInput{
		TableName: &d.serversTable,
		Item:      item,
	}

	// For DDB put is update for our purposes
	_, err = d.aws.DDBPutItem(input)

	return err
}

func (d *ddbStateStore) ListDaemons() ([]*Daemon, error) {
	d.logger.Debug("ListDaemons: start")
	input := &dynamodb.ScanInput{
		TableName: &d.serversTable,
	}

	d.logger.Debugf("ListDaemons: scanning %v", d.serversTable)

	output, err := d.aws.DDBScan(input)
	if err != nil {
		d.logger.Warnf("Error scanning dynamo db: %v", err)
		return nil, err
	}

	ret := make([]*Daemon, 0)

	for _, item := range output.Items {
		daemon := &Daemon{}
		err := dynamodbattribute.UnmarshalMap(item, daemon)
		if err != nil {
			return nil, err
		}
		ret = append(ret, daemon)
	}

	d.logger.Debug("ListDaemons: end")

	return ret, nil
}

// Users
func (d *ddbStateStore) AddUser(user *User) error {
	item, err := dynamodbattribute.MarshalMap(user)
	if err != nil {
		return err
	}

	condition := "twitch_user_id <> :twitch_user_id"

	input := &dynamodb.PutItemInput{
		TableName:           &d.usersTable,
		Item:                item,
		ConditionExpression: &condition,
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":twitch_user_id": &dynamodb.AttributeValue{S: &user.TwitchUserID},
		},
	}

	_, err = d.aws.DDBPutItem(input)

	return err
}

func (d *ddbStateStore) GetUser(ID string) (*User, error) {
	input := &dynamodb.GetItemInput{
		TableName: &d.usersTable,
		Key: map[string]*dynamodb.AttributeValue{
			"twitch_user_id": &dynamodb.AttributeValue{S: &ID},
		},
	}

	item, err := d.aws.DDBGetItem(input)

	if err != nil {
		return nil, err
	}

	if len(item.Item) == 0 {
		return nil, errors.New("User Not Found")
	}

	ret := &User{}

	err = dynamodbattribute.UnmarshalMap(item.Item, ret)

	if err != nil {
		return nil, err
	}

	return ret, nil
}

func (d *ddbStateStore) RemoveUser(ID string) error {
	input := &dynamodb.DeleteItemInput{
		TableName: &d.usersTable,
		Key: map[string]*dynamodb.AttributeValue{
			"twitch_user_id": &dynamodb.AttributeValue{S: &ID},
		},
	}

	_, err := d.aws.DDBDeleteItem(input)

	return err
}

func (d *ddbStateStore) UpdateUser(user *User) error {
	item, err := dynamodbattribute.MarshalMap(user)
	if err != nil {
		return err
	}

	input := &dynamodb.PutItemInput{
		TableName: &d.usersTable,
		Item:      item,
	}

	// For DDB put is update for our purposes
	_, err = d.aws.DDBPutItem(input)

	return err
}

func (d *ddbStateStore) ListUsers() ([]*User, error) {
	d.logger.Debug("ListUsers: start")
	input := &dynamodb.ScanInput{
		TableName: &d.usersTable,
	}

	d.logger.Debugf("ListUsers: scanning %v", d.usersTable)

	output, err := d.aws.DDBScan(input)
	if err != nil {
		d.logger.Warnf("Error scanning dynamo db: %v", err)
		return nil, err
	}

	ret := make([]*User, 0)

	for _, item := range output.Items {
		user := &User{}
		err := dynamodbattribute.UnmarshalMap(item, user)
		if err != nil {
			return nil, err
		}
		ret = append(ret, user)
	}

	d.logger.Debug("ListUsers: end")

	return ret, nil
}

// RTMPSource
func (d *ddbStateStore) AddRTMPSource(source *RTMPSource) error {
	item, err := dynamodbattribute.MarshalMap(source)
	if err != nil {
		return err
	}

	condition := "source_id <> :source_id"

	input := &dynamodb.PutItemInput{
		TableName:           &d.rtmpSourcesTable,
		Item:                item,
		ConditionExpression: &condition,
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":source_id": &dynamodb.AttributeValue{S: &source.SourceID},
		},
	}

	_, err = d.aws.DDBPutItem(input)

	return err
}

func (d *ddbStateStore) GetRTMPSource(ID string) (*RTMPSource, error) {
	input := &dynamodb.GetItemInput{
		TableName: &d.rtmpSourcesTable,
		Key: map[string]*dynamodb.AttributeValue{
			"source_id": &dynamodb.AttributeValue{S: &ID},
		},
	}

	item, err := d.aws.DDBGetItem(input)

	if err != nil {
		return nil, err
	}

	if len(item.Item) == 0 {
		return nil, errors.New("RTMP Source Not Found")
	}

	ret := &RTMPSource{}

	err = dynamodbattribute.UnmarshalMap(item.Item, ret)

	if err != nil {
		return nil, err
	}

	return ret, nil
}

func (d *ddbStateStore) GetRTMPSourceByRTMPKey(RTMPKey string) (*RTMPSource, error) {
	filter := "rtmp_key = :rtmp_key"
	index := "RtmpKeyIndex"

	input := &dynamodb.QueryInput{
		TableName:              &d.rtmpSourcesTable,
		IndexName:              &index,
		KeyConditionExpression: &filter,
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":rtmp_key": &dynamodb.AttributeValue{S: &RTMPKey},
		},
	}

	output, err := d.aws.DDBQuery(input)

	if err != nil {
		d.logger.Warnf("Error querying dynamo db: %v", err)
		return nil, err
	}

	if len(output.Items) == 0 {
		return nil, fmt.Errorf("No RTMP Source found with RTMP Key: %v", RTMPKey)
	}

	if len(output.Items) > 1 {
		return nil, fmt.Errorf("Found %v RTMP Sources with RTMP Key %v - We expect there to be only ONE", len(output.Items), RTMPKey)
	}

	ret := &RTMPSource{}

	err = dynamodbattribute.UnmarshalMap(output.Items[0], ret)

	if err != nil {
		return nil, err
	}

	return ret, nil
}

func (d *ddbStateStore) RemoveRTMPSource(ID string) error {
	input := &dynamodb.DeleteItemInput{
		TableName: &d.rtmpSourcesTable,
		Key: map[string]*dynamodb.AttributeValue{
			"source_id": &dynamodb.AttributeValue{S: &ID},
		},
	}

	_, err := d.aws.DDBDeleteItem(input)

	return err
}

func (d *ddbStateStore) UpdateRTMPSource(source *RTMPSource) error {
	item, err := dynamodbattribute.MarshalMap(source)
	if err != nil {
		return err
	}

	input := &dynamodb.PutItemInput{
		TableName: &d.rtmpSourcesTable,
		Item:      item,
	}

	// For DDB put is update for our purposes
	_, err = d.aws.DDBPutItem(input)

	return err
}

func (d *ddbStateStore) ListRTMPSources() ([]*RTMPSource, error) {
	d.logger.Debug("ListRTMPSources: start")
	input := &dynamodb.ScanInput{
		TableName: &d.rtmpSourcesTable,
	}

	d.logger.Debugf("ListRTMPSources: scanning %v", d.rtmpSourcesTable)

	output, err := d.aws.DDBScan(input)
	if err != nil {
		d.logger.Warnf("Error scanning dynamo db: %v", err)
		return nil, err
	}

	ret := make([]*RTMPSource, 0)

	for _, item := range output.Items {
		source := &RTMPSource{}
		err := dynamodbattribute.UnmarshalMap(item, source)
		if err != nil {
			return nil, err
		}
		ret = append(ret, source)
	}

	d.logger.Debug("ListRTMPSources: end")

	return ret, nil
}
