package db

import (
	"time"
)

type MoonlightDB interface {
	ListInstancesByServer(serverID string) ([]*Instance, error)
	ListInstancesByRtmpKey(rtmpKey string) ([]*Instance, error)
	ListInstancesByTwitchUserID(twitchUserID string) ([]*Instance, error)
	ListInstances() ([]*Instance, error)
	AddInstance(instance *Instance) error
	GetInstance(id string) (*Instance, error)
	UpdateInstance(instance *Instance) error
	RemoveInstance(id string) error
	AddDaemon(daemon *Daemon) error
	ListDaemons() ([]*Daemon, error)
	GetDaemon(serverId string) (*Daemon, error)
	UpdateDaemon(daemon *Daemon) error
	RemoveDaemon(id string) error
	AddUser(user *User) error
	ListUsers() ([]*User, error)
	GetUser(id string) (*User, error)
	UpdateUser(user *User) error
	RemoveUser(id string) error
	AddRTMPSource(user *RTMPSource) error
	ListRTMPSources() ([]*RTMPSource, error)
	GetRTMPSource(id string) (*RTMPSource, error)
	GetRTMPSourceByRTMPKey(RTMPKey string) (*RTMPSource, error)
	UpdateRTMPSource(user *RTMPSource) error
	RemoveRTMPSource(id string) error
}

type InstanceStatus string

const (
	InstanceNew            InstanceStatus = "NEW"
	InstanceAwaitingServer InstanceStatus = "AWAITING_SERVER"
	InstanceSlotReserved   InstanceStatus = "SLOT_RESERVED"
	InstanceRegistered     InstanceStatus = "REGISTERED"
	InstanceRunning        InstanceStatus = "RUNNING"
	InstanceHealthy        InstanceStatus = "HEALTHY"
	InstanceUnhealthy      InstanceStatus = "UNHEALTHY"
	InstanceError          InstanceStatus = "ERRORED"
)

type DaemonStatus string

const (
	DaemonReady DaemonStatus = "READY"
)

type Instance struct {
	ID              string         `json:"instance_id"`
	ServerID        string         `json:"server_id"`
	TwitchUserID    string         `json:"twitch_user_id"`
	RTMPKey         string         `json:"rtmp_key"`
	OBSScenesJSON   string         `json:"obs_scenes_json"`
	DockerImageName string         `json:"docker_image_name"`
	Status          InstanceStatus `json:"status"`
	RTMPMuted       bool           `json:"rtmp_muted"`
	Heartbeat       string         `json:"heartbeat"`
	LastUpdated     time.Time      `json:"last_updated"`
	LastStreamed    time.Time      `json:"last_streamed"`
}

type Daemon struct {
	ID           string       `json:"server_id"`
	RPCURL       string       `json:"rpc_url"`
	Status       DaemonStatus `json:"status"`
	LastUpdated  time.Time    `json:"last_updated"`
	NumInstances int32        `json:"num_instances"`
}

type User struct {
	TwitchUserID  string `json:"twitch_user_id"`
	Active        bool   `json:"active"`
	RTMPSource    string `json:"rtmp_source"`
	OBSScenesJSON string `json:"obs_scenes_json"`
}

type RTMPSource struct {
	SourceID     string `json:"source_id"`
	RTMPKey      string `json:"rtmp_key"`
	InternalAddr string `json:"internal_addr"`
}
