package external

import (
	"code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	"code.justin.tv/event-engineering/moonlight-api/pkg/aws"
	awsBackend "code.justin.tv/event-engineering/moonlight-api/pkg/aws/backend"
	controlRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/control"
	"code.justin.tv/event-engineering/moonlight-api/pkg/twitch"
	"encoding/json"
	"fmt"
	"log"
	"net/http"
)

const AuthorizerHeaderName = "X-Authorizer"

type handler struct {
	mux      *http.ServeMux
	logger   logging.Logger
	internal controlRPC.InternalController
	twitch   *twitch.Client
}

// New will create a new external API to be called by lambda
func New(internalLambdaARN string, ab awsBackend.Client, twitch *twitch.Client, logger logging.Logger) http.Handler {
	awsClient := aws.New(ab, logger)
	lambdaTransport := lambda.NewClient(awsClient.GetLambdaClient(), internalLambdaARN)
	internal := controlRPC.NewInternalControllerJSONClient("https://www.doesnt.matter.because.its.actually.lambda.transport.twitch.tv", lambdaTransport)

	handler := &handler{
		mux:      http.NewServeMux(),
		logger:   logger,
		internal: internal,
		twitch:   twitch,
	}

	handler.mux.HandleFunc("/info", handler.Info)
	handler.mux.HandleFunc("/init", handler.Init)
	handler.mux.HandleFunc("/settings", handler.UpdateOBSConfig)
	handler.mux.HandleFunc("/status", handler.Status)
	handler.mux.HandleFunc("/live", handler.Live)
	handler.mux.HandleFunc("/mute", handler.Mute)
	handler.mux.HandleFunc("/current-scene", handler.SetScene)

	return handler.mux
}

// Separating the internal type we use to make decisions from the external type that we send back in the /info request
// to prevent accidentally adding new properties that might be sensitive back to the user
type userInfo struct {
	UserID    string `json:"user_id"`
	UserName  string `json:"user_name"`
	TokenUsed string `json:"token_used"`
}

type Error struct {
	Message string `json:"message"`
}

func getUserInfo(request *http.Request) (userInfo, error) {
	user := userInfo{}

	// Information about who the user is is passed in via the API gateway lambda authoriser using the X-Authorizer header
	// It should hopefully be impossible to get here without a valid X-Authorizer header if I've done this right :)
	authoriser := request.Header.Get(AuthorizerHeaderName)

	err := json.Unmarshal([]byte(authoriser), &user)

	if err != nil {
		return userInfo{}, fmt.Errorf("Failed to unmarshall authoriser %v | %v", err, authoriser)
	}

	if user.UserID == "" || user.UserName == "" {
		return userInfo{}, fmt.Errorf("Invalid user info %v | %v", user.UserID, user.UserName)
	}

	return user, nil
}

func sendResponse(writer http.ResponseWriter, statusCode int, body interface{}) error {
	writer.WriteHeader(statusCode)

	writer.Header().Set("Access-Control-Allow-Origin", "*")

	bytes, err := json.Marshal(body)
	if err != nil {
		return err
	}

	_, err = writer.Write(bytes)
	if err != nil {
		return err
	}

	return nil
}

func handleError(writer http.ResponseWriter, statusCode int, message string) {
	err := sendResponse(writer, statusCode, Error{Message: message})

	if err != nil {
		log.Println(err)
	}
}
