package external

import (
	controlRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/control"
	"encoding/json"
	"github.com/twitchtv/twirp"
	"net/http"
)

func (s *handler) Init(writer http.ResponseWriter, request *http.Request) {
	user, err := getUserInfo(request)
	if err != nil {
		s.logger.Warnf("Error authenticating user %v", err)
		handleError(writer, http.StatusUnauthorized, "")
		return
	}

	resp, err := s.internal.InitialiseSession(request.Context(), &controlRPC.InitialiseSessionReq{
		TwitchUserId: user.UserID,
	})

	if err != nil {
		if twerr, ok := err.(twirp.Error); ok {
			switch twerr.Code() {
			case twirp.NotFound:
				handleError(writer, http.StatusNotFound, twerr.Msg())
			}
		}

		s.logger.Warnf("Error executing InitialiseSession %v", err)
		handleError(writer, http.StatusInternalServerError, "")
		return
	}

	if resp.GetInstanceId() == "" {
		s.logger.Warnf("Error executing InitialiseSession, no instance ID", err)
		handleError(writer, http.StatusInternalServerError, "No instance created")
		return
	}

	sendResponse(writer, http.StatusOK, InitialiseSessionResp{
		Config: StreamConfig{
			Scenes: resp.Scenes,
		},
		InstanceInfo: StatusResp{
			Status:        resp.InstanceInfo.GetStatus(),
			RTMPMuted:     resp.InstanceInfo.GetRtmpMuted(),
			HeartbeatJson: resp.InstanceInfo.GetHeartbeatJson(),
		},
	})
}

func (s *handler) UpdateOBSConfig(writer http.ResponseWriter, request *http.Request) {
	user, err := getUserInfo(request)
	if err != nil {
		s.logger.Warnf("Error authenticating user %v", err)
		handleError(writer, http.StatusUnauthorized, "")
		return
	}

	decoder := json.NewDecoder(request.Body)

	obsConfig := &UpdateOBSConfigReq{}
	err = decoder.Decode(obsConfig)

	if err != nil {
		s.logger.Warnf("Error deserialising body %v", err)
		handleError(writer, http.StatusBadRequest, "")
	}

	resp, err := s.internal.UpdateOBSConfig(request.Context(), &controlRPC.UpdateOBSConfigReq{
		TwitchUserId:      user.UserID,
		MainOverlayUrl:    obsConfig.MainOverlayURL,
		MainOverlayWidth:  obsConfig.MainOverlayWidth,
		MainOverlayHeight: obsConfig.MainOverlayHeight,
		MainOverlayX:      obsConfig.MainOverlayX,
		MainOverlayY:      obsConfig.MainOverlayY,
	})

	if err != nil {
		s.logger.Warnf("Error executing UpdateOBSConfig %v", err)
		handleError(writer, http.StatusInternalServerError, "")
		return
	}

	if !resp.GetSuccess() {
		s.logger.Warnf("Error executing UpdateOBSConfig %v", err)
		handleError(writer, http.StatusBadRequest, resp.GetMessage())
		return
	}

	sendResponse(writer, http.StatusOK, UserInfo{
		user.UserID,
		user.UserName,
	})
}
