package jobrunner

import (
	db "code.justin.tv/event-engineering/moonlight-api/pkg/db"
	daemonRPC "code.justin.tv/event-engineering/moonlight-daemon/pkg/rpc"
	"context"
	"errors"
	"sort"
)

const MaxInstancesPerServer = 20
const AllocateServerMaxReceiveCount = 8

func (c *client) AllocateServer(ctx context.Context, body AllocateServerMessage) error {
	instance, err := c.db.GetInstance(body.InstanceID)
	if err != nil {
		return err
	}

	// The slot will be reserved on the server for (1 minute)??? it is expected that the instance
	// be initialised on the server within this timeframe or the slot will be deallocated
	serverID, err := c.allocateServer(ctx, instance.ID)
	if err != nil {
		// If this errors there are no available servers, in this scenario we wait for a new server to be spun up
		// returning an error results in the SQS message not being deleted so it will be picked up again after the visibility timeout
		// we need some good tracking around this so that we can alarm if this happens
		return err
	}

	instance.Status = db.InstanceRegistered
	instance.ServerID = serverID

	err = c.db.UpdateInstance(instance)
	if err != nil {
		return err
	}

	return nil
}

func (c *client) allocateServer(ctx context.Context, instanceID string) (string, error) {
	servers, err := c.db.ListDaemons()

	if err != nil {
		return "", err
	}

	// Sort by NumInstances desc
	sort.Slice(servers, func(i, j int) bool {
		return servers[i].NumInstances > servers[j].NumInstances
	})

	// Load up a single server to max before allocating to another server, this helps with autoscaling efficiently
	for _, server := range servers {
		if server.NumInstances < MaxInstancesPerServer {
			err := c.attemptAllocation(ctx, instanceID, server)
			if err != nil {
				c.logger.Warnf("Failed to allocate server with ID %v | %v", server.ID, err)
				continue
			}
			return server.ID, nil
		}
	}

	return "", errors.New("No servers available for allocation")
}

func (c *client) attemptAllocation(ctx context.Context, instanceID string, server *db.Daemon) error {
	client := daemonRPC.NewDaemonJSONClient(server.RPCURL, c.daemonHttpClient)
	_, err := client.AllocateInstance(ctx, &daemonRPC.AllocateInstanceReq{
		InstanceId: instanceID,
	})

	if err != nil {
		return err
	}

	return nil
}
