package jobrunner

import (
	db "code.justin.tv/event-engineering/moonlight-api/pkg/db"
	daemonRPC "code.justin.tv/event-engineering/moonlight-daemon/pkg/rpc"
	"context"
	"time"
)

const maxInstanceIdleTime = time.Duration(time.Minute * 10)

func (c *client) CheckConsistency(ctx context.Context) error {
	c.logger.Info("Checking Consistency")

	instances, err := c.db.ListInstances()
	if err != nil {
		c.logger.Warnf("Failed to retrieve instances %v", err)
		return err
	}

	servers, err := c.db.ListDaemons()
	if err != nil {
		c.logger.Warnf("Failed to retrieve servers %v", err)
		return err
	}

	now := time.Now()

	for _, instance := range instances {
		// Kill instances that haven't streamed for 10m
		if instance.LastStreamed.Add(maxInstanceIdleTime).Before(now) {
			c.logger.Infof("Attempting to remove instance with ID %v due to idle timeout", instance.ID)

			var server *db.Daemon
			for _, sv := range servers {
				if sv.ID == instance.ServerID {
					server = sv
					break
				}
			}

			if server != nil {
				client := daemonRPC.NewDaemonJSONClient(server.RPCURL, c.daemonHttpClient)
				_, stopErr := client.StopInstance(ctx, &daemonRPC.StopInstanceReq{
					InstanceId: instance.ID,
				})
				if stopErr != nil {
					c.logger.Warnf("Failed to stop instance %v with error %v", instance.ID, stopErr)
					return err
				}
			} else {
				c.logger.Warnf("Instance %v was allocated to non-existent server %v", instance.ID, instance.ServerID)
			}

			err = c.db.RemoveInstance(instance.ID)
			if err != nil {
				c.logger.Warnf("Failed to remove instance %v from db with error %v", instance.ID, err)
				return err
			}
		}
	}

	return nil
}
