package jobrunner

import (
	"context"
	"fmt"
	"sort"

	daemonRPC "code.justin.tv/event-engineering/moonlight-daemon/pkg/rpc"
	"github.com/aws/aws-sdk-go/service/autoscaling"
)

func (c *client) CheckScaling(ctx context.Context, autoscalingGroupName string) error {
	c.logger.Info("Checking Scaling")

	servers, err := c.db.ListDaemons()
	if err != nil {
		c.logger.Warnf("Failed to retrieve servers %v", err)
		return err
	}

	// Sort by NumInstances desc
	sort.Slice(servers, func(i, j int) bool {
		return servers[i].NumInstances > servers[j].NumInstances
	})

	var availableSlots int

	numServers := len(servers)

	for i, daemon := range servers {
		c.logger.Infof("Daemon %v has %v instances", daemon.ID, daemon.NumInstances)
		availableThisServer := (AllocateServerMaxReceiveCount - int(daemon.NumInstances))

		// If this is the last server in the list (least loaded)
		if numServers == i+1 {
			c.logger.Infof("Available slots: %v, total slots %v", availableThisServer+availableSlots, AllocateServerMaxReceiveCount*numServers)

			if float64(availableThisServer+availableSlots)/float64(AllocateServerMaxReceiveCount) <= float64(0.25) {
				c.logger.Infof("Percent available %v", float64(availableThisServer+availableSlots)/float64(AllocateServerMaxReceiveCount))
				// Spin up a new server
				desiredCapacity := int64(numServers + 1)
				_, err := c.aws.AutoScalingSetDesiredCapacity(&autoscaling.SetDesiredCapacityInput{
					AutoScalingGroupName: &autoscalingGroupName,
					DesiredCapacity:      &desiredCapacity,
				})

				if err != nil {
					c.logger.Warnf("Failed to set desired capacity %v", err)
					return err
				}

				break
			} else if numServers > 1 && daemon.NumInstances == 0 && (float64(availableSlots)/float64(AllocateServerMaxReceiveCount) > float64(0.25)) {
				c.logger.Infof("Percent available if server terminated %v", float64(availableSlots)/float64(AllocateServerMaxReceiveCount))
				// Terminate the daemon
				client := daemonRPC.NewDaemonJSONClient(daemon.RPCURL, c.daemonHttpClient)
				resp, err := client.Terminate(ctx, &daemonRPC.TerminateDaemonReq{})

				if err != nil {
					return fmt.Errorf("Error when calling terminate daemon %v with error %v", daemon.ID, resp.Message)
				}

				if !resp.Success {
					return fmt.Errorf("Could not terminate daemon %v reason: %v", daemon.ID, resp.Message)
				}

				// Terminate instance
				shouldDecrementDesiredCapacity := true
				_, err = c.aws.AutoScalingTerminateInstance(&autoscaling.TerminateInstanceInAutoScalingGroupInput{
					InstanceId:                     &daemon.ID,
					ShouldDecrementDesiredCapacity: &shouldDecrementDesiredCapacity,
				})
				if err != nil {
					c.logger.Warnf("Failed to terminate instance %v with error %v", daemon.ID, err)
					return err
				}
				break
			}
		}

		availableSlots += availableThisServer
	}

	return nil
}
